/***************************************************************************
 *   Copyright (C) 2005 - 2007 by                                          *
 *      Christian Muehlhaeuser, Last.fm Ltd <chris@last.fm>                *
 *      Erik Jaelevik, Last.fm Ltd <erik@last.fm>                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02111-1307, USA.          *
 ***************************************************************************/

// Got to have this first so Winsock doesn't moan about multiple defines
#include "container.h"
#include "lastfmapplication.h"
#include "configwizard.h"
#include "containerutils.h"
#include "logindialog.h"
#include "usermetadata.h"
#include "logger.h"
#include "Settings.h"
#include "playercommands.h"
#include "version.h"

#include <QTimer>

#ifdef WIN32
    #include <windows.h>
#endif


LastFmApplication::LastFmApplication( int& argc, char** argv ) :
        QApplication( argc, argv ),
        m_endSessionEmitted( false )
{
    // We're doing this here because the version.h file is updated just before
    // each build of container so if we include it directly in lastfmtools it
    // will always lag one version number behind.
    The::settings().setVersion( LASTFM_CLIENT_VERSION );
    The::settings().saveAppPath();

    initLogger();
	initTranslator();
	registerMetaTypes();

    setQuitOnLastWindowClosed( false );

  #ifndef WIN32
    setWindowIcon( QIcon( dataPath( "icons/as.ico" ) ) );
  #endif

    // This is needed so that relative paths will work on Windows regardless
    // of where the app is launched from.
    QDir::setCurrent( applicationDirPath() );

    // run init() as soon as event loop starts
    QTimer::singleShot( 0, this, SLOT(init()) );

    QString version = The::settings().version();
    LOGL( 1, "App version: " << version );
}


void
LastFmApplication::init()
{
    // Need to save the state from before we run the wizard as the wizard will change it
    bool firstRunBeforeWizard = The::settings().isFirstRun();
    
    if (The::settings().isFirstRun())
    {
        LOG( 3, "First run, launching config wizard\n" );
        QFile( savePath( "mediadevice.db" ) ).remove();

        ConfigWizard wiz( NULL, ConfigWizard::Login );
        if ( wiz.exec() == QDialog::Rejected )
        {
            // If user cancels config wizard, we need to exit
            Container::instance().actualClose( false );
            return;
        }
    }

    // Do we have a current user?
    QString currentUser = The::settings().currentUsername();
    bool doLogin = false;
    if ( currentUser.isEmpty() )
    {
        LOG( 3, "No current user\n" );
        doLogin = true;
    }
    else
    {
        doLogin = !The::settings().user( currentUser ).rememberPass();
    }

    if ( doLogin && !firstRunBeforeWizard )
    {
        LOG( 3, "Ask for login\n" );

        LoginDialog login( NULL, LoginWidget::LOGIN, currentUser );
        if ( login.exec() != QDialog::Accepted )
        {
            Container::instance().actualClose( false );
            return;
        }
    }
    else
        // the initServices must be called, but the logindialog causes it to 
        // be called because the The::settings()::save() function causes it to be called
        // when it emits a signal
        Container::instance().initServices();

    // This is needed for the app to shut down properly when asked by Windows
    // to shut down.
    connect( this, SIGNAL(endSession()), this, SLOT(shutdown()) );

    if ( !arguments().contains( "-tray" ) && !arguments().contains( "--tray" ) )
    {
        Container::instance().show();
    }
}


void
LastFmApplication::shutdown()
{
    // Shuts down without asking the user (e.g. for a system-event).
    Container::instance().actualClose( false );
}


void
LastFmApplication::initLogger()
{
    Logger& logger = Logger::GetLogger();

    logger.Init( savePath( "container.log" ), false );
    logger.SetLevel( Logger::Debug );
}


void
LastFmApplication::initTranslator()
{
    Q_DEBUG_BLOCK;
    
    QString langCode;

    #ifdef HIDE_RADIO
		langCode = "jp";
	#else

        #ifndef Q_WS_MAC
            // First check settings, but never on mac, always get that from the system
            langCode = The::settings().appLanguage();
        #endif

        if ( langCode.isEmpty() )
        {
            // If none found, use system locale
            QLocale::Language qtLang = QLocale::system().language();
            langCode = qtLanguageToLfmLangCode( qtLang );
            The::settings().setAppLanguage( langCode );
            
            LOGL( 3, "System locale: " << langCode );
        }
    
	#endif

    setLanguage( langCode );
    installTranslator( &m_translatorApp );
    installTranslator( &m_translatorQt );
}


void
LastFmApplication::setLanguage( QString langCode )
{
    LOGL( 3, "Setting language to: " << langCode );

    m_lang = langCode;

    // Discards previously loaded translations
    m_translatorApp.load( dataPath( "i18n/lastfm_%1" ).arg( langCode ) );
    m_translatorQt.load( dataPath( "i18n/qt_%1" ).arg( langCode ) );
}


void
LastFmApplication::registerMetaTypes()
{
    // This is needed so we can pass MetaData objects as signal/slot params
    // with queued connections.
    qRegisterMetaType<TrackInfo>( "TrackInfo" );
    qRegisterMetaType<MetaData>( "MetaData" );
    qRegisterMetaType<UserMetaData>( "UserMetaData" );
    qRegisterMetaType<CPlayerCommand>( "CPlayerCommand" );
    qRegisterMetaType<RadioError>( "RadioError" );
    qRegisterMetaType<RadioState>( "RadioState" );
}


#ifdef WIN32
bool
LastFmApplication::winEventFilter( MSG * msg, long * result )
{
    /*
    typedef struct MSG {
        HWND        hwnd;
        UINT        message;
        WPARAM      wParam;
        LPARAM      lParam;
        DWORD       time;
        POINT       pt;
    }*/

    // This message is sent by Windows when we're being shut down as part
    // of a Windows shutdown. Don't want to just minimise to tray so we
    // emit a special endSession signal to Container. It can get sent
    // several times though so we must guard against emitting the signal
    // more than once.
    if ( msg->message == WM_QUERYENDSESSION )
    {
        if ( !m_endSessionEmitted )
        {
            m_endSessionEmitted = true;
            emit endSession();
        }
        *result = 1; // tell Windows it's OK to shut down
        return true; // consume message
    }

    return false; // let Qt handle it
}
#endif // WIN32
