/***************************************************************************
 *   Copyright (C) 2005 - 2007 by                                          *
 *      Christian Muehlhaeuser, Last.fm Ltd <chris@last.fm>                *
 *      Erik Jaelevik, Last.fm Ltd <erik@last.fm>                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02111-1307, USA.          *
 ***************************************************************************/

#include "containerutils.h" //MD5Digest
#include "lastfm_common.h"
#include "Settings.h"
#include <QCoreApplication>
#include <QDir>
#include <QFile>
#include <QFileInfo>

#ifdef Q_WS_MAC
#include <ApplicationServices/ApplicationServices.h>
#endif



/******************************************************************************
 * UserSettings
 ******************************************************************************/

UserSettings::UserSettings( const QString& username )
        : m_username( username )
{}


QString
UserSettings::password() const
{
    return MyQSettings( this ).value( "Password" ).toString();
}


void
UserSettings::setPassword( QString password )
{
    if (!password.isEmpty() && password != "********")
    {
        password = MD5Digest( password.toUtf8() );
        MyQSettings( this ).setValue( "Password", password );
        emit userChanged( username() );
    }
}


UserIconColour
UserSettings::icon() const
{
    MyQSettings s( this );

    // This will return eRed if there is no entry. Don't want that.
    if (s.contains( "Icon" ))
        return (UserIconColour) s.value( "Icon" ).toInt();
    else
        return eNone; 
}


void
UserSettings::setIcon( UserIconColour colour )
{
    MyQSettings( this ).setValue( "Icon", static_cast<int>(colour) );
    emit userChanged( username() );
}


/// Written as int for backwards compatibility with the MFC Audioscrobbler
bool
UserSettings::isLogToProfile() const
{
    return static_cast<bool>(MyQSettings( this ).value( "LogToProfile", 1 ).toInt());
}


/// Written as int for backwards compatibility with the MFC Audioscrobbler
void
UserSettings::setLogToProfile( bool state )
{
    MyQSettings( this ).setValue( "LogToProfile", static_cast<int>(state) );
    emit userChanged( username() );
}


bool
UserSettings::isDiscovery() const
{
    return MyQSettings( this ).value( "DiscoveryEnabled", false ).toBool();
}


void
UserSettings::setDiscovery( bool state )
{
    MyQSettings( this ).setValue( "DiscoveryEnabled", state );
    emit userChanged( username() );
}


bool
UserSettings::sidebarEnabled() const
{
    return MyQSettings( this ).value( "SidebarEnabled", false ).toBool();
}


void
UserSettings::setSidebarEnabled( bool state )
{
    MyQSettings( this ).setValue( "SidebarEnabled", state );
    emit userChanged( username() );
}


/// Written as int for backwards compatibility with the MFC Audioscrobbler
bool
UserSettings::rememberPass() const
{
    return (bool) MyQSettings( this ).value( "RememberPass", true ).toInt();
}


/// Written as int for backwards compatibility with the MFC Audioscrobbler
void
UserSettings::setRememberPass( bool remember )
{
    MyQSettings( this ).setValue( "RememberPass", int(remember) );
    emit userChanged( username() );
}


void
UserSettings::setResumePlayback( bool enabled )
{
    MyQSettings( this ).setValue( "resumeplayback", enabled ? "1" : "0" );
    emit userChanged( username() ); 
}


void
UserSettings::setResumeStation( StationUrl station )
{
    MyQSettings( this ).setValue( "resumestation", station );
    emit userChanged( username() );
}


void
UserSettings::addRecentStation( const Station& station )
{
    MyQSettings s( this );

    QList<Station> stations = recentStations();
    
    // remove duplicates
    for (int i = 0; i < stations.count(); ++i)
        if (stations[i].url() == station.url())
            stations.removeAt( i-- );
    
    stations.prepend( station );
    
    s.remove( "RecentStations" );
    
    s.beginGroup( "RecentStations" );
    int j = stations.count();
    while (j--)
        s.setValue( QString::number( j ), stations[j].url() );
    s.endGroup();
    
    s.setValue( "StationNames/" + station.url(), station.name() );
    s.sync();

    emit userChanged( username() );
    emit historyChanged();
}


void
UserSettings::removeRecentStation( int n )
{
    MyQSettings s( this );
    
    QString const N = QString::number( n );
    
    s.beginGroup( "RecentStations" );
    QString const url = s.value( N ).toString();
    s.remove( N );
    
    // now renumber in correct order (maps are auto-sorted by key)
    QMap<int, QString> urls;
    foreach (QString key, s.childKeys())
        urls[key.toInt()] = s.value( key ).toString();
    
    remove( "" ); //current group
            
    int i = 0;
    foreach (QString url, urls)
        s.setValue( QString::number( i++ ), url );
    s.endGroup();
    
    s.remove( "StationNames/" + url );
    s.sync();
    
    emit userChanged( username() );
    emit historyChanged();
}


void
UserSettings::clearRecentStations( bool emitting )
{
    MyQSettings( this ).remove( "RecentStations" );

    //TODO needed still?
    if (emitting)
        emit historyChanged();
}


QList<Station>
UserSettings::recentStations()
{
    MyQSettings s( this );

    s.beginGroup( "RecentStations" );
    QStringList const keys = s.childKeys();
    s.endGroup();

    QMap<int, Station> stations;
    foreach (QString key, keys) {
        Station station;
        station.setUrl( s.value( "RecentStations/" + key ).toString() );
        station.setName( s.value( "StationNames/" + station.url() ).toString() );
        stations[key.toInt()] = station;
    }

    return stations.values();
}


void
UserSettings::save()
{
    MyQSettings( this ).sync();
}


void
UserSettings::setExcludedDirs( QStringList dirs )
{
    MyQSettings( this ).setValue( "ExclusionDirs", dirs );
    emit userChanged( username() );
}


QStringList
UserSettings::excludedDirs() const
{
    return MyQSettings( this ).value( "ExclusionDirs" ).toStringList();
}


void
UserSettings::setIncludedDirs( QStringList dirs )
{
    MyQSettings( this ).setValue( "InclusionDirs", dirs );
    emit userChanged( username() );
}


QStringList
UserSettings::includedDirs() const
{
    return MyQSettings( this ).value( "InclusionDirs" ).toStringList();
}

void
UserSettings::setMetadataEnabled( bool enabled )
{
    MyQSettings( this ).setValue( "DownloadMetadata", enabled );
    emit userChanged( username() );
}

bool
UserSettings::metadataEnabled()
{
    return MyQSettings( this ).value( "DownloadMetadata", true ).toBool();
}

void
UserSettings::setCrashReportingEnabled( bool enabled )
{
    MyQSettings( this ).setValue( "ReportCrashes", enabled );
    emit userChanged( username() );
}

bool
UserSettings::crashReportingEnabled()
{
    return MyQSettings( this ).value( "ReportCrashes", true ).toBool();
}

void
UserSettings::setScrobblePoint( int scrobblePoint )
{
    MyQSettings( this ).setValue( "ScrobblePoint", scrobblePoint );
    emit userChanged( username() );
}

int
UserSettings::scrobblePoint()
{
    return MyQSettings( this ).value( "ScrobblePoint", Defaults::kScrobblePoint ).toInt();
}

void
UserSettings::setTrackFrameClockMode( bool trackTimeEnabled )
{
    MyQSettings( this ).setValue( "TrackFrameShowsTrackTime", trackTimeEnabled );
    emit userChanged( username() );
}

bool
UserSettings::trackFrameClockMode()
{
    return MyQSettings( this ).value( "TrackFrameShowsTrackTime", true ).toBool();
}

void 
UserSettings::setLaunchWithMediaPlayer( bool en )
{
    MyQSettings( this ).setValue( "LaunchWithMediaPlayer", en );
    emit userChanged( username() );
}

bool
UserSettings::launchWithMediaPlayer()
{
    return MyQSettings( this ).value( "LaunchWithMediaPlayer", true ).toBool();
}


/******************************************************************************
 *   Settings
 ******************************************************************************/

Settings::Settings( QObject* parent ) :
        QObject( parent ),
        m_nullUser( "" )
{
    LOGL( 3, "Initialising Settings Service" );

    #ifndef WIN32
        QSettings new_config;
        
        if (!QFile( new_config.fileName() ).exists())
        {
            //attempt to upgrade settings object from old and broken location
            foreach (QString const name, QStringList() << "Client" << "Users" << "Plugins" << "MediaDevices")
            {
                QSettings old_config( QSettings::IniFormat, QSettings::UserScope, "Last.fm", name );
                old_config.setFallbacksEnabled( false );
                
                if (!QFile::exists( old_config.fileName() ))
                    continue;
                
                foreach (QString const key, old_config.allKeys()) {
                    if (name != "Client")
                        //Client now becomes [General] group as this makes most sense
                        new_config.beginGroup( name );
                    new_config.setValue( key, old_config.value( key ) );
                    #ifndef QT_NO_DEBUG
                    if (name != "Client") // otherwise qWarning and aborts
                    #endif
                    new_config.endGroup();
                }
                
                new_config.sync();
                
                QFile f( old_config.fileName() );
                f.remove();
                QFileInfo( f ).dir().rmdir( "." ); //safe as won't remove a non empty dir
            }
        }
    #endif
}

void
Settings::saveAppPath()
{
    // Can't do this in ctor because other executables (like CrashReporter) might
    // create a Settings object!
    QSettings().setValue( "Path", QCoreApplication::applicationFilePath() );
}

UserSettings&
Settings::user( QString username ) const
{
    Q_ASSERT( username != "" );
    
    UserSettings *user = findChild<UserSettings*>( username );

    if (!user) {
        user = new UserSettings( username );
        user->setParent( const_cast<Settings*>(this) );
        user->setObjectName( username );
        connect( user, SIGNAL(userChanged( QString )), SLOT(userChanged( QString )) );
    }
    
    return *user;
}

UserSettings&
Settings::currentUser()
{
    return currentUsername() == ""
            ? m_nullUser
            : user( currentUsername() );
}

void
Settings::setCurrentUsername( QString username ) 
{
    UsersSettings().setValue( CURRENT_USER_KEY, username );
    
    emit userSettingsChanged( currentUser() );
    emit userSwitched( currentUser() );
}

bool
Settings::deleteUser( QString username )
{
    if (isExistingUser( username ))
    {
        delete &user( username );
        UsersSettings().remove( username );
        
        return true;
    }
    else
        return false;
}


int
Settings::externalSoundSystem()
{
    int externalSystem = -1;
    #ifdef WIN32
    externalSystem = 1;
    #endif
    #ifdef Q_WS_X11
    externalSystem = 2;
    #endif
    #ifdef Q_WS_MAC
    externalSystem = 1;
    #endif

    return externalSystem;
}



void
Settings::setDontAsk( const QString op, bool value )
{
    QSettings().setValue( op + "DontAsk", value );
}

bool
Settings::isDontAsk( const QString op ) const
{
    return QSettings().value( op + "DontAsk" ).toBool();
}

void
Settings::setAppLanguage( QString langCode )
{
    QSettings().setValue( "AppLanguage", langCode );
}

QString
Settings::appLanguage() const
{
    return QSettings().value( "AppLanguage" ).toString();
}


void
Settings::save( bool restartConnection, bool restartAudio )
{
    QSettings().sync();

    if (restartConnection)
        emit doReconnect();

    if (restartAudio)
        emit doResetAudio();
}


QStringList
Settings::allPlugins( bool withVersions )
{
    PluginsSettings s;
    QStringList plugins;

    foreach (QString group, s.childGroups()) {
        s.beginGroup( group );
        QString name = s.value( "Name" ).toString();
        if ( withVersions )
        {
            QString version = s.value( "Version" ).toString();
            plugins += name + ' ' + tr("plugin, version") + ' ' + version;
        }
        else
        {
            plugins += name;
        }
        s.endGroup();
    }

    return plugins;
}


QString
Settings::pluginVersion( QString id )
{
    Q_ASSERT( !id.isEmpty() );

    return PluginsSettings().value( id + "/Version" ).toString();
}


QString
Settings::pluginPlayerPath( QString id )
{
    Q_ASSERT( !id.isEmpty() );

    return PluginsSettings().value( id + "/PlayerPath" ).toString();
}


void
Settings::setPluginPlayerPath( QString id,
                               QString path )
{
    Q_ASSERT( !id.isEmpty() );

    PluginsSettings().setValue( id + "/PlayerPath", path );
}


QStringList
Settings::allMediaDevices()
{
    MediaDeviceSettings s;
    return s.childGroups();
}


QString
Settings::mediaDeviceUser( QString uid ) const
{
    MediaDeviceSettings s;
    s.beginGroup( uid );
    return s.value( "user" ).toString();
}


void
Settings::addMediaDevice( QString uid, QString username )
{
    MediaDeviceSettings s;
    s.beginGroup( uid );
    s.setValue( "user", username );
    s.sync();
}


void
Settings::removeMediaDevice( QString uid )
{
    MediaDeviceSettings s;
    s.beginGroup( uid );
    s.remove( "user" );
    s.sync();
}


UserIconColour
Settings::getFreeColour()
{
    UsersSettings s;
    QList<int> unused;

    // Fill it with all colours
    for (int i = 0; i < 5; ++i)
    {
        unused.push_back(i);
    }

    // Remove the ones in use
    foreach (QString username, s.childGroups())
    {
        UserIconColour col = UserSettings( username ).icon();

        if (col != eNone)
            unused.removeAll( int(col) );

        if (unused.isEmpty()) {
            LOG( 2, "We ran out of colours, returning random\n" );
            return static_cast<UserIconColour>(rand() % 5);
        }
    }

    return static_cast<UserIconColour>(unused.front());
}


void
Settings::userChanged( QString username )
{
    if (username == currentUsername())
        emit userSettingsChanged( currentUser() );
}


QString
Settings::localizedHostName() const
{
    QString const l = appLanguage();

    if (l == "en") return "www.last.fm"; //first as optimisation
    if (l == "pt") return "www.lastfm.com.br";
    if (l == "cn") return "cn.last.fm";
        
    QStringList simple_hosts = QStringList() 
            << "fr" << "it" << "de" << "es" << "pl"
            << "ru" << "jp" << "se" << "tr";
    if (simple_hosts.contains( l ))
        return "www.lastfm." + l;
        
    // else default to english site
    return "www.last.fm";
}
