/***************************************************************************
 *   Copyright (C) 2005 - 2007 by                                          *
 *      Christian Muehlhaeuser, Last.fm Ltd <chris@last.fm>                *
 *      Erik Jaelevik, Last.fm Ltd <erik@last.fm>                          *
 *      Max Howell, Last.fm Ltd. <max@last.fm>                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02111-1307, USA.          *
 ***************************************************************************/

#ifndef SETTINGS_H
#define SETTINGS_H

#include "last.fm.h"
#include "lastfm_common.h"
#include "lastfmtools_common.h"
#include "StationUrl.h"
#include <QMap>
#include <QPoint>
#include <QSettings>
#include <QSize>
#include <QStringList>

#if 1
#  define CURRENT_USER_KEY "CurrentUser"
#else
#  define CURRENT_USER_KEY "DebugBuildCurrentUser"
#endif


/**
 * Note we operate on separate QSettings objects always to be thread-safe.
 * Note the silly classes for each group are because you can't copy QSettings objects
 * so we couldn't just return one through a function, which would be neater.
 */

#ifdef WIN32
    class LocalMachineSettings : public QSettings
    {
    public:
        LocalMachineSettings() :
                QSettings( "HKEY_LOCAL_MACHINE\\Software\\Last.fm\\Client",
                           QSettings::NativeFormat )
        {}
    };
    
    /** Due to historical reasons, we store windows settings "wrongly", but
      * migrate settings on other platforms to the "correct" location. To make
      * the code all the same though we use this class below and macro it to
      * the QSettings token */
    class WindowsQSettings : public QSettings
    {
    public:
        WindowsQSettings() : QSettings( "HKEY_CURRENT_USER\\Software\\Last.fm\\Client",
                                        QSettings::NativeFormat )
        {}
    };
    
    #define QSettings WindowsQSettings
#else
    typedef QSettings LocalMachineSettings;
#endif // WIN32


class PluginsSettings : public LocalMachineSettings
{
public:
    PluginsSettings()
    {
        beginGroup( "Plugins" );
    }
};


/** NOTE don't confuse this with UserSettings later! */
class UsersSettings : public QSettings
{
public:
    UsersSettings()
    {
        beginGroup( "Users" );
    }
};


class MediaDeviceSettings : public QSettings
{
public:
    MediaDeviceSettings()
    {
        beginGroup( "MediaDevices" );
    }
};


class MCEXP UserSettings : public QObject
{
    Q_OBJECT
    
    class MyQSettings : public UsersSettings
    {
    public:
        MyQSettings( UserSettings const *const s )
        {
            beginGroup( s->username() );
        }
    };

public:
    UserSettings( const QString& username );

    bool isNull() const { return m_username.isEmpty(); }

    QString username() const { return m_username; }

    QString password() const;
    void setPassword( QString password );

    UserIconColour icon() const;
    void setIcon( UserIconColour colour );

    bool isLogToProfile() const;
    void setLogToProfile( bool state );
    void toggleLogToProfile() { setLogToProfile( !isLogToProfile() ); }

    bool isDiscovery() const;
    void setDiscovery( bool state );

    bool sidebarEnabled() const;
    void setSidebarEnabled( bool state );

    int  lastTagType( int type = 0 ) const { return MyQSettings( this ).value( "lasttagtype", type ).toInt(); }
    void setLastTagType( int type )        { MyQSettings( this ).setValue( "lasttagtype", type ); }
    
    int  lastRecommendType()        const { return MyQSettings( this ).value( "lastrecommendtype", 1 ).toInt(); /*Track is default*/ }
    void setLastRecommendType( int type ) { MyQSettings( this ).setValue( "lastrecommendtype", type ); }

    bool rememberPass() const;
    void setRememberPass( bool remember );

    bool resumePlayback() const { return MyQSettings( this ).value( "resumeplayback", 0 ).toInt() == 1; }
    void setResumePlayback( bool enabled );

    StationUrl resumeStation() const
    {
        return StationUrl( MyQSettings( this ).value( "resumestation" ).toString() );
    }
    
    void setResumeStation( StationUrl station );
    
    void addRecentStation( const class Station& );
    void removeRecentStation( int list_index );
    void clearRecentStations( bool emit_signal );
    QList<Station> recentStations();

    QStringList excludedDirs() const;
    void setExcludedDirs( QStringList dirs );
    
    QStringList includedDirs() const;
    void setIncludedDirs( QStringList dirs );

    bool metadataEnabled();
    void setMetadataEnabled( bool enabled );

    bool crashReportingEnabled();
    void setCrashReportingEnabled( bool enabled );

    int scrobblePoint();
    void setScrobblePoint( int scrobblePoint );

    QString lastRecommendee() const { return MyQSettings( this ).value( "LastRecommendee" ).toString(); }
    void    setLastRecommendee( QString v ) { MyQSettings( this ).setValue( "LastRecommendee", v ); }
    
    int personalTagsListSortOrder() const { return MyQSettings( this ).value( "PersonalTagListSortOrder" ).toInt(); };
    void setPersonalTagsListSortOrder( int sortOrder ) { MyQSettings( this ).setValue( "PersonalTagListSortOrder", sortOrder ); };
    int publicTagsListSortOrder() const { return MyQSettings( this ).value( "PublicTagListSortOrder" ).toInt(); };
    void setPublicTagsListSortOrder( int sortOrder ) { MyQSettings( this ).setValue( "PublicTagListSortOrder", sortOrder ); };

    int sideBarTagsSortOrder() const { return MyQSettings( this ).value( "SideBarTagsSortOrder" ).toInt(); };
    void setSideBarTagsSortOrder( int sortOrder ) { MyQSettings( this ).setValue( "SideBarTagsSortOrder", sortOrder ); };
    int sideBarNeighbourSortOrder() const { return MyQSettings( this ).value( "SideBarNeighbourSortOrder" ).toInt(); };
    void setSideBarNeighbourSortOrder( int sortOrder ) { MyQSettings( this ).setValue( "SideBarNeighbourSortOrder", sortOrder ); };
    
    void setTrackFrameClockMode( bool trackTimeEnabled );
    bool trackFrameClockMode();

    void setLaunchWithMediaPlayer( bool en );
    bool launchWithMediaPlayer();

signals:
    void userChanged( QString username );
    void historyChanged();

public slots:
    void save();

private:
    QString m_username;
};


class MCEXP Settings : public QObject
{
    Q_OBJECT

    Settings( QObject* parent );

    friend Settings &The::settings();

public:
    QString version() const
    {
        // ask Erik if you wonder why we store this in the configuration
        return QSettings().value( "Version", "unknown" ).toString();
    }

    void setVersion( QString v ) { QSettings().setValue( "Version", v ); }

    QByteArray containerGeometry() const { return QSettings().value( "MainWindowGeometry" ).toByteArray(); }
    void setContainerGeometry( QByteArray state ) { QSettings().setValue( "MainWindowGeometry", state ); }
    
    Qt::WindowState containerWindowState() const { return (Qt::WindowState) QSettings().value( "MainWindowState" ).toInt(); }
    void setContainerWindowState( int state ) { QSettings().setValue( "MainWindowState", state ); }

    QByteArray splitterState() const { return QSettings().value( "splitterState" ).toByteArray(); }
    void setSplitterState( QByteArray state ) { QSettings().setValue( "splitterState", state ); }

    QStringList allUsers() const { return UsersSettings().childGroups(); }
    void setCurrentUsername( QString username );
    QString currentUsername() const { return UsersSettings().value( CURRENT_USER_KEY ).toString(); }    

    UserSettings& user( QString username ) const;
    UserSettings& currentUser();

    bool deleteUser( QString username );
    bool isExistingUser( QString username ) const { return UsersSettings().contains( username + "/Password" ); }

    QStringList allPlugins( bool withVersions = true );

    // Returns "" if QSettings().value not found
    QString pluginVersion( QString pluginId );
    QString pluginPlayerPath( QString pluginId );
    void setPluginPlayerPath( QString pluginId, QString path );

    QStringList allMediaDevices();

    /// Returns "" if QSettings().value not found
    QString mediaDeviceUser( QString uid ) const;
    void addMediaDevice( QString uid, QString username );
    void removeMediaDevice( QString uid );

    int  volume()     const { return QSettings().value( "volume", 50 ).toInt(); }
    void setVolume( int v ) { QSettings().setValue( "volume", v ); }

    int  soundCard()     const { return QSettings().value( "soundcard", 0 ).toInt(); }
    void setSoundCard( int v ) { QSettings().setValue( "soundcard", v ); }

    int  soundSystem()     const { return QSettings().value( "soundsystem", 0 ).toInt(); }
    void setSoundSystem( int v ) { QSettings().setValue( "soundsystem", v ); }

    bool isBufferManagedAutomatically() const { return QSettings().value( "BufferManagedAutomatically", 1 ).toBool(); }
    void setBufferManagedAutomatically( bool v ) { QSettings().setValue( "BufferManagedAutomatically", v ); }

    int  httpBufferSize()     const { return QSettings().value( "HttpBufferSize", Constants::kHttpBufferMinSize ).toInt(); }
    void setHttpBufferSize( int v ) { QSettings().setValue( "HttpBufferSize", v ); }

    int externalSoundSystem();

//////
    bool isUseProxy()    const { return QSettings().value( "ProxyEnabled" ).toInt() == 1; }
    void setUseProxy( bool v ) { QSettings().setValue( "ProxyEnabled", v ? "1" : "0" ); }

    QString getProxyHost()      const { return QSettings().value( "ProxyHost" ).toString(); }
    void    setProxyHost( QString v ) { QSettings().setValue( "ProxyHost", v ); }

    int  getProxyPort()  const { return QSettings().value( "ProxyPort" ).toInt(); }
    void setProxyPort( int v ) { QSettings().setValue( "ProxyPort", v ); }

    QString getProxyUser()      const { return QSettings().value( "ProxyUser" ).toString(); }
    void    setProxyUser( QString v ) { QSettings().setValue( "ProxyUser", v ); }

    QString getProxyPassword()      const { return QSettings().value( "ProxyPassword" ).toString(); }
    void    setProxyPassword( QString v ) { QSettings().setValue( "ProxyPassword", v ); }

//////
    QString browser()            const { return QSettings().value( "Browser" ).toString(); }
    void setBrowser( QString browser ) { QSettings().setValue( "Browser", browser ); }

    // This is a string for legacy reasons
    bool isFirstRun() const { return LocalMachineSettings().value( "FirstRun", "1" ).toBool(); }
    void setFirstRunDone()  { LocalMachineSettings().setValue( "FirstRun", "0" ); }

    // This is a string for legacy reasons
    bool isBootstrapDone() const { return QSettings().value( "BootStrapDone", "0" ).toBool(); }
    void setBootstrapDone()      { QSettings().setValue( "BootStrapDone", "1" ); }

    bool  isDontAsk( QString operation_name ) const;
    void setDontAsk( QString operation_name, bool );

    /// Use one of our pre-defined 2-letter language codes
    QString appLanguage() const;
    void setAppLanguage( QString langCode ); 

    /// here because it is contextually related, but well, should be somewhere else I expect
    /// @returns www.last.fm, www.lastfm.com.br, etc.
    //NOTE there is no localizedUrl() function as last.fm urls require special encoding
    // for certain path fragments, thus you have to construct them yourself! :(
    //NOTE we don't prefix http:// as usage in the app varies
    QString localizedHostName() const;
    
    UserIconColour getFreeColour();

    void saveAppPath();

public slots:
    void save( bool restartConnection = false, bool restartAudio = false );

signals:
    void doReconnect();
    void doResetAudio();

    /// Some property belonging to the current user changed
    void userSettingsChanged( UserSettings& user );

    /// The current user changed to a different user
    void userSwitched( UserSettings& newUser );

private slots:
    void userChanged( QString username );

private:
    // because we have to return by reference :(
    UserSettings m_nullUser;
};


class UserQSettings : public UsersSettings
{
public:
    UserQSettings( UserSettings* user )
    {
        beginGroup( user->username() );
    }
};


namespace The
{
    inline Settings &settings()
    {
        //TODO maybe better to have a static instantiate() function
        // thus we lose the need for a mutex

        static QMutex mutex;
        static Settings* settings = 0;

        QMutexLocker locker( &mutex );

        if (!settings)
        {
            settings = QCoreApplication::instance()->findChild<Settings*>( "Settings-Instance" );
            if (!settings)
            {
                settings = new Settings( QCoreApplication::instance() );
                settings->setObjectName( "Settings-Instance" );
            }
        }
        return *settings;
    }

    inline UserSettings &user( QString username )
    {
        return The::settings().user( username );
    }

    inline UserSettings &currentUser()
    {
        return The::settings().currentUser();
    }

    inline QString currentUsername()
    {
        return The::settings().currentUsername();
    }
}


class CurrentUserSettings : public UserQSettings
{
public:
    CurrentUserSettings() : UserQSettings( &The::currentUser() )
    {}
};

#endif
