/***************************************************************************
 *   Copyright (C) 2005 - 2007 by                                          *
 *      Christian Muehlhaeuser, Last.fm Ltd <chris@last.fm>                *
 *      Erik Jaelevik, Last.fm Ltd <erik@last.fm>                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02111-1307, USA.          *
 ***************************************************************************/

#include "containerutils.h"
#include "interfaces/MediaDeviceInterface.h"

#include <QtGui>
#include <QHttp>
#include <QObject>
#include <QProcess>

//#include "logger.h"
#include "md5/md5.h"

#ifdef WIN32
    //#define _WINSOCKAPI_   /* Prevent inclusion of winsock.h in windows.h */
    #include "windows.h"
    #include "shfolder.h"
#endif

#ifdef Q_WS_MAC
	#include <Carbon/Carbon.h>
#endif

static MediaDeviceInterface*     g_iTunesDevice = NULL;

QString MD5Digest( const char *token )
{
    md5_state_t md5state;
    unsigned char md5pword[16];

    md5_init( &md5state );
    md5_append( &md5state, (unsigned const char *)token, (int)strlen( token ) );
    md5_finish( &md5state, md5pword );

    char tmp[33];
    strncpy( tmp, "\0", sizeof(tmp) );
    for ( int j = 0; j < 16; j++ )
    {
        char a[3];
        sprintf( a, "%02x", md5pword[j] );
        tmp[2*j] = a[0];
        tmp[2*j+1] = a[1];
    }

    return QString::fromAscii( tmp );
}


QString dataPath( QString file )
{
    return QApplication::applicationDirPath() + "/data/" + file;
}


QString savePath( QString file )
{
    QString prefix;
    QString midfix;

    prefix = QApplication::applicationDirPath();
    #ifdef WIN32
        if ( ( QSysInfo::WindowsVersion & QSysInfo::WV_DOS_based ) == 0 )
        {
            // Use this for non-DOS-based Windowses
            char acPath[MAX_PATH];
            HRESULT h = SHGetFolderPathA( NULL, CSIDL_LOCAL_APPDATA | CSIDL_FLAG_CREATE,
                                          NULL, 0, acPath );
            if ( h == S_OK )
            {
                prefix = QString::fromLocal8Bit( acPath );
                prefix.append( "/Last.fm/Client" );
            }
            else
            {
                //LOG( 1, "Couldn't get Local Settings save path\n" );
            }
        }
	
    #elif defined(Q_WS_MAC)
		prefix = getApplicationSupportFolder() + "/Last.fm";
    
    #elif defined(Q_WS_X11)
        prefix = QDir::home().filePath( ".local/share/Last.fm" );
    
    #else
		prefix = QApplication::applicationDirPath();
    #endif

    QDir d( prefix );
    d.mkpath( prefix );

    return d.filePath( file );
}


QString pluginPath( QString name )
{
    QString dirPath;
    #ifdef WIN32
        // Hack to get it working with VS2005
        dirPath = qApp->applicationDirPath();
    #else
        dirPath = qApp->applicationDirPath() + "/services";
    #endif

    #ifndef QT_NO_DEBUG
        dirPath += "/debug";
        name += DEBUG_SUFFIX;
    #endif

    QDir servicesDir( dirPath );
    QString fileName = SERVICE_PREFIX + name + LIB_EXTENSION;

    return servicesDir.absoluteFilePath( fileName );
}


QString cachePath()
{
	#ifdef Q_WS_MAC
		QString appSupportFolder = getApplicationSupportFolder();
		QDir cacheDir( appSupportFolder + "/Last.fm/Cache" );
		
		if (!cacheDir.exists())
		{
			cacheDir.mkpath( appSupportFolder + "/Last.fm/Cache" );
		}
		
		return cacheDir.path() + "/";
	#else
		return savePath( "cache/" );
	#endif
}


static void loadServiceError( QString name )
{
    QMessageBox::critical( 0,
        QCoreApplication::translate( "Container", "Error" ),
        QCoreApplication::translate( "Container", "Couldn't load service: %1. The application won't be able to start." ).arg( name ) );

    QCoreApplication::exit( 1 );
}


static QObject* loadService( QString name )
{
    QString path = pluginPath( name );

	qDebug() << "Loading service: " << name << "at" << path;

    QObject* plugin = QPluginLoader( path ).instance();

    if ( plugin == NULL )
    {
        loadServiceError( name );
        return NULL;
    }

    return plugin;
}


/*
    We're taking a slight risk by not guarding these calls with mutexes because
    they can in theory be accessed by different threads. However, since they
    only get loaded once at startup, this is not a problem in practice. If
    we decide to synchronise access in the future we'd need a separate mutex
    for each getter function, as otherwise we'd end up with same thread
    deadlocks.
*/

MediaDeviceInterface* iTunesDevice()
{
    if ( g_iTunesDevice == NULL )
    {
    #ifndef LINUX
        QObject* obj = loadService( "itunesdevice" );
        g_iTunesDevice = qobject_cast<MediaDeviceInterface*>( obj );
    #endif
    }

    return g_iTunesDevice;
}


QString QHttpStateToString(int state)
{
    switch (state)
    {
        case QHttp::Unconnected: return QCoreApplication::translate( "WebService", "No connection." );
        case QHttp::HostLookup: return QCoreApplication::translate( "WebService", "Looking up host..." );
        case QHttp::Connecting: return QCoreApplication::translate( "WebService", "Connecting..." );
        case QHttp::Sending: return QCoreApplication::translate( "WebService", "Sending request..." );
        case QHttp::Reading: return QCoreApplication::translate( "WebService", "Downloading." );
        case QHttp::Connected: return QCoreApplication::translate( "WebService", "Connected." );
        case QHttp::Closing: return QCoreApplication::translate( "WebService", "Closing connection..." );
        default: return QString();
    }
}


QString qtLanguageToLfmLangCode( QLocale::Language qtLang )
{
    switch ( qtLang )
    {
        case QLocale::English:    return "en";
        case QLocale::French:     return "fr";
        case QLocale::Italian:    return "it";
        case QLocale::German:     return "de";
        case QLocale::Spanish:    return "es";
        case QLocale::Portuguese: return "pt";
        case QLocale::Polish:     return "pl";
        case QLocale::Russian:    return "ru";
        case QLocale::Japanese:   return "jp";
        case QLocale::Chinese:    return "cn";
        case QLocale::Swedish:    return "sv";
        case QLocale::Turkish:    return "tr";
        default:                  return "en";
    }
}


QString lfmLangCodeToIso639( QString code )
{
    if ( code == "jp" ) return "ja";
    if ( code == "cn" ) return "zh";

    return code;
}


bool installHelperApp()
{
    #ifdef Q_WS_MAC
        QDir( QDir::homePath() + "/Library" ).mkdir( "LaunchAgents" );
        QString dst = QDir::homePath() + "/Library/LaunchAgents/fm.last.lastfmhelper.plist";
        QString helperApp = qApp->applicationDirPath() + "/LastFMHelper";

        #ifndef QT_NO_DEBUG
            helperApp += "_debug";
        #endif

        QStringList unload;
        unload << "unload" << dst;
        QProcess::execute( QString( "/bin/launchctl" ), unload );

        QFile::remove( dst );
        QFile r( qApp->applicationDirPath() + "/fm.last.lastfmhelper.plist" );
        QFile f( dst );

        r.open( QIODevice::ReadOnly | QIODevice::Text );
        f.open( QIODevice::WriteOnly | QIODevice::Text );
        QString s = QString( r.readAll() ).replace( "%HELPERAPP%", helperApp );
        f.write( s.toLocal8Bit() );
        f.close();
        r.close();

        QStringList load;
        load << "load" << dst;
        QProcess::execute( QString( "/bin/launchctl" ), load );
    #endif

    return true;
}

QString pathToCachedCopy( QString cacheKey )
{
    QString keyMd5 = MD5Digest( qPrintable( cacheKey ) );
    return cachePath() + keyMd5;
}


#include "metadata.h"

namespace The
{
    MetaData const &currentMetaData()
    {
        //HACK! better system for this stuff
    
        static MetaData data;
    
        return data;
    }
}

#ifdef Q_WS_MAC
	QString 
	getApplicationSupportFolder()
	{
		std::string outString;
		
		OSErr err;
		
		short vRefNum = 0;
		StrFileName fileName;
		fileName[0] = 0;
		long dirId;
		err = ::FindFolder(kOnAppropriateDisk, kApplicationSupportFolderType,
						   kDontCreateFolder, &vRefNum, &dirId);
		if (err != noErr) return "";
		
		// Now we have a vRefNum and a dirID - but *not* an Unix-Path as string.
		// Lets make one based from this:
		
		// create a FSSpec...
		FSSpec fsspec;
		err = ::FSMakeFSSpec(vRefNum, dirId, NULL, &fsspec);
		if (err != noErr) return "";
		
		// ...and build an FSRef based on thes FSSpec.
		FSRef fsref;
		err = ::FSpMakeFSRef(&fsspec, &fsref);
		if (err != noErr) return "";
		
		// ...then extract the Unix Path as a C-String from the FSRef
		unsigned char path[512];
		err = ::FSRefMakePath(&fsref, path, 512);
		if (err != noErr) return "";
		
		// ...and copy this into the result.
		outString = (const char*)path;

		return QDir::homePath() + QString::fromStdString( outString );
	}
#endif
