/***************************************************************************
 *   Copyright (C) 2005 - 2007 by                                          *
 *      Christian Muehlhaeuser, Last.fm Ltd <chris@last.fm>                *
 *      Erik Jaelevik, Last.fm Ltd <erik@last.fm>                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02111-1307, USA.          *
 ***************************************************************************/

#include <QtGui>
#include <QPixmap>

#include "browserthread.h"
#include "metadataextension.h"
#include "container.h"
#include "containerutils.h" //dataPath savePath
#include "Settings.h"
#include "http.h"
#include "utils.h"
#include "logger.h"
#include "WebService/Request.h"

// These are pixel sizes
#ifdef Q_WS_MAC
    static const int k_trackFontSize = 18;
    static const int k_artistFontSize = 14;
    static const int k_standardFontSize = 12;
#else
    static const int k_trackFontSize = 17;
    static const int k_artistFontSize = 15;
    static const int k_standardFontSize = 11;
#endif

static const QColor k_greyFontColour = QColor( 0x98, 0x98, 0x98 );
static const QColor k_userActionFontColour = QColor( 0x8e, 0x9e, 0xb3 );
static const QColor k_notListeningFontColour = QColor( 0xAF, 0xAF, 0xAF );

static const int k_marginTop = 22;
static const int k_marginLeft = 22;
static const int k_marginRight = 22;
static const int k_marginBottom = 22;

static const int k_column = 16; // width between images and text
static const int k_row = 30; // height between track and artist sections

static const int k_spaceAfterCaption = 4; // between colon and label


MetaDataExtension::MetaDataExtension()
        : m_widget( 0 )
        , m_coverArtistLoader( new Http( this ) )
        , m_coverAlbumLoader( new Http( this ) )
        , m_haveTrack( false )
        , m_requestsReset( false )
        , m_offline( false )
        , m_metadataFetched( NothingFetched )
        , m_alwaysFetchMetaData( false )
{
    LOG( 3, "Initialising MetaData Extension\n" );

    // Redirects on the site will take care of shuffling users along to the
    // right language version.
    m_urlBase = "http://" + The::settings().localizedHostName();

    connect( m_coverArtistLoader, SIGNAL( dataAvailable( QByteArray ) ),
             this,                SLOT( coverArtistLoaded( QByteArray ) ) );
    connect( m_coverAlbumLoader, SIGNAL( dataAvailable( QByteArray ) ),
             this,               SLOT( coverAlbumLoaded( QByteArray ) ) );

    m_tuning_in_timer = new QTimer( this );
    m_tuning_in_timer->setInterval( 6000 );
    m_tuning_in_timer->setSingleShot( true );
    connect( m_tuning_in_timer, SIGNAL(timeout()), SLOT(onDelayedTuningIn()) );
}


QString
MetaDataExtension::name() const
{
    return QString( "MetaData Extension" );
}


QString
MetaDataExtension::version() const
{
    return QString( "0.0.1" );
}

QWidget*
MetaDataExtension::gui()
{
    if ( !m_widget )
        initGui();

    return &m_stack;
}

void
MetaDataExtension::initGui()
{
    LOG( 3, "Initialising MetaData GUI\n" );


    #ifndef Q_WS_MAC
    // Watermark
    WatermarkWidget* watermarkwidget = new MetaDataWidget( this, m_parent );
    watermarkwidget->setWatermark( dataPath( "watermark.png" ) );
    m_widget = watermarkwidget;
    #else
    m_widget = new QWidget( m_parent );
    #endif
    
    ui.setupUi( m_widget );
    
    // Set default font on everything as it doesn't work on the Mac otherwise
    QFont defFont = m_widget->font();
    defFont.setPixelSize( k_standardFontSize );
    ui.buyTrackLabel->setFont( defFont ); 
    ui.dateAndLabelLabel->setFont( defFont );
    ui.totalCaption->setFont( defFont );
    ui.numTracksLabel->setFont( defFont );
    ui.releasedCaption->setFont( defFont );
    ui.buyAlbumLabel->setFont( defFont ); 
    ui.wikiLabel->setFont( defFont );
    ui.wikiLink->setFont( defFont ); 
    ui.tagNoTagsLabel->setFont( defFont );
    ui.tagLink->setFont( defFont );
    
    // Track font
    defFont.setPixelSize( k_trackFontSize );
    defFont.setBold( true );
    ui.trackLabel->setFont( defFont );

    // Artist, album & about font
    defFont.setPixelSize( k_artistFontSize );
    ui.byCaption->setFont( defFont );
    ui.artistLabel->setFont( defFont );
    ui.albumLabel->setFont( defFont );
    ui.aboutLabel->setFont( defFont );

    // Num listeners font
    defFont = m_widget->font();
    defFont.setBold( true );
    defFont.setPixelSize( k_standardFontSize );
    QPalette defPal = m_widget->palette();
    defPal.setColor( QPalette::Text, k_greyFontColour );
    ui.numListenersLabel->setFont( defFont );
    ui.numListenersLabel->setPalette( defPal );

    // Item types for drag labels
    ui.trackLabel->setItemType( ItemTrack );
    ui.artistLabel->setItemType( ItemArtist );
    ui.albumLabel->setItemType( ItemAlbum );
    ui.aboutLabel->setItemType( ItemArtist );
    ui.tagsLabel->setItemType( ItemTag );
    ui.similarLabel->setItemType( ItemArtist );
    ui.topFansLabel->setItemType( ItemUser );

    // No word wrap on one-liners
    ui.trackLabel->setWordWrap( false );
    ui.artistLabel->setWordWrap( false );
    ui.buyTrackLabel->setWordWrap( false );
    ui.albumLabel->setWordWrap( false );
    ui.dateAndLabelLabel->setWordWrap( false );
    ui.numTracksLabel->setWordWrap( false );
    ui.buyAlbumLabel->setWordWrap( false );
    ui.aboutLabel->setWordWrap( false );
    ui.numListenersLabel->setWordWrap( false );
    ui.wikiLink->setWordWrap( false );

    // Set up comma-separated fields with headers
    defFont.setPixelSize( k_standardFontSize );
    ui.tagsLabel->setHeader( tr( "Tags:" ), defFont );
    ui.similarLabel->setHeader( tr( "Similar artists:" ), defFont );
    ui.topFansLabel->setHeader( tr( "Top listeners on Last.fm:" ), defFont );
    ui.tagsLabel->setCommaSeparated( true );
    ui.similarLabel->setCommaSeparated( true );
    ui.topFansLabel->setCommaSeparated( true );

    ui.tagNoTagsLabel->setText( tr( "No one has tagged this artist yet." ) );
    ui.tagLink->setText( tr( "Tag this artist..." ) );
    
    // Hand cursor on image hover
    ui.artistPic->setHoverCursor( QCursor( Qt::PointingHandCursor ) );
    ui.albumPic->setHoverCursor( QCursor( Qt::PointingHandCursor ) );

    // Load icons
    bool success = m_buyTrack.load( dataPath( "icons/buy_track.png" ) );
    success &= m_buyAlbum.load( dataPath( "icons/buy_album.png" ) );
    if ( !success )
    {
        LOGL( 1, "Failed to load buy icons" );
    }

    // Set up web links
    ui.buyTrackIcon->setImages( m_buyTrack, m_buyTrack, m_buyTrack, m_buyTrack );
    ui.buyTrackIcon->setHoverCursor( Qt::PointingHandCursor );
    //ui.buyTrackLabel->setLinkColor( QColor( 0, 0, 0 ) );
    ui.buyTrackLabel->setHighlightedColor( QColor( 0, 0, 0 ) );
    ui.buyTrackLabel->setSelectedColor( QColor( 100, 100, 100 ) );

    ui.buyAlbumIcon->setImages( m_buyAlbum, m_buyAlbum, m_buyAlbum, m_buyAlbum );
    ui.buyAlbumIcon->setHoverCursor( Qt::PointingHandCursor );
    //ui.buyAlbumLabel->setLinkColor( QColor( 0, 0, 0 ) );
    ui.buyAlbumLabel->setHighlightedColor( QColor( 0, 0, 0 ) );
    ui.buyAlbumLabel->setSelectedColor( QColor( 100, 100, 100 ) );

    ui.writeWikiButton->setImages( dataPath( "buttons/" ),
                             "action_edit.png",
                             "action_edit_down.png",
                             "action_edit_hover.png",
                             "action_edit.png" );
    ui.writeWikiButton->hide();

    ui.tagButton->setImages( dataPath( "buttons/" ),
                             "action_tag.png",
                             "action_tag_down.png",
                             "action_tag_hover.png",
                             "action_tag.png" );

    ui.wikiLink->setHighlightedColor( QColor( 0, 0, 0 ) );
    ui.wikiLink->setSelectedColor( QColor( 100, 100, 100 ) );

    ui.tagLink->setHighlightedColor( k_userActionFontColour );
    ui.tagLink->setSelectedColor( QColor( 100, 100, 100 ) );
    
    // Not listening screen
    WatermarkWidget* notPlaying = new WatermarkWidget( m_parent );
    ui_notPlaying.setupUi( notPlaying );
    notPlaying->setWatermark( dataPath( "watermark.png" ) );
    
    QPalette p = notPlaying->palette();
    p.setColor( QPalette::Window, QColor( 0xff, 0xff, 0xff ) );
    p.setColor( QPalette::Base, QColor( 0xff, 0xff, 0xff ) );
    notPlaying->setPalette( p );

    QPixmap logo( dataPath( "logo.png" ) );
    ui_notPlaying.logoLabel->setPixmap( logo );

    m_spinnerMovie.setFileName( dataPath( "progress.mng" ) );
    m_spinnerMovie.start();
    ui_notPlaying.spinnerLabel->setMovie( &m_spinnerMovie );

    p = ui_notPlaying.messageLabel->palette();
    p.setColor( QPalette::WindowText, k_notListeningFontColour );
    p.setColor( QPalette::Text, k_notListeningFontColour );
    ui_notPlaying.messageLabel->setPalette( p );

    // Scroll area
    m_scrollArea = new MetaDataScrollArea( this, &m_stack );

    p = m_scrollArea->palette();
    p.setColor( QPalette::Window, Qt::white );
    m_scrollArea->setPalette( p );
    m_scrollArea->setHorizontalScrollBarPolicy( Qt::ScrollBarAsNeeded );
    m_scrollArea->setVerticalScrollBarPolicy( Qt::ScrollBarAsNeeded );
    m_scrollArea->setFrameStyle( QFrame::NoFrame );
    m_scrollArea->setWidgetResizable( true );
    m_scrollArea->setWidget( m_widget );

    m_stack.addWidget( m_scrollArea );
    m_stack.addWidget( notPlaying );

    applyMarginAndSpacing();

//////
    connect( ui.artistPic,     SIGNAL(clicked()), SLOT(artistPicClicked()) );
    connect( ui.albumPic,      SIGNAL(clicked()), SLOT(albumPicClicked()) );
    connect( ui.buyTrackIcon,  SIGNAL(clicked()), SLOT(buyTrackIconClicked()) );
    connect( ui.buyAlbumIcon,  SIGNAL(clicked()), SLOT(buyAlbumIconClicked()) );    
    
    connect( ui.writeWikiButton, SIGNAL(clicked()), ui.wikiLink, SLOT(openURL()) );
    
    connect( ui.tagButton, SIGNAL(clicked()),        SIGNAL(tagButtonClicked()) );
    connect( ui.tagLink,   SIGNAL(leftClickedURL()), SIGNAL(tagButtonClicked()) );

    connect( ui.trackLabel,   SIGNAL(urlHovered( const QString& )), SIGNAL(urlHovered( const QString&) ) );
    connect( ui.artistLabel,  SIGNAL(urlHovered( const QString& )), SIGNAL(urlHovered( const QString&) ) );
    connect( ui.albumLabel,   SIGNAL(urlHovered( const QString& )), SIGNAL(urlHovered( const QString&) ) );
    connect( ui.aboutLabel,   SIGNAL(urlHovered( const QString& )), SIGNAL(urlHovered( const QString&) ) );
    connect( ui.tagsLabel,    SIGNAL(urlHovered( const QString& )), SIGNAL(urlHovered( const QString&) ) );
    connect( ui.similarLabel, SIGNAL(urlHovered( const QString& )), SIGNAL(urlHovered( const QString&) ) );
    connect( ui.topFansLabel, SIGNAL(urlHovered( const QString& )), SIGNAL(urlHovered( const QString&) ) );

    connect( ui.buyTrackLabel, SIGNAL(enteredURL( const QString& )), SIGNAL(urlHovered( const QString&) ) );
    connect( ui.buyAlbumLabel, SIGNAL(enteredURL( const QString& )), SIGNAL(urlHovered( const QString&) ) );
    connect( ui.wikiLink,      SIGNAL(enteredURL( const QString& )), SIGNAL(urlHovered( const QString&) ) );
    
    connect( ui.buyTrackLabel, SIGNAL(leftURL()), SLOT(urlLeft()) );
    connect( ui.buyAlbumLabel, SIGNAL(leftURL()), SLOT(urlLeft()) );
    connect( ui.wikiLink,      SIGNAL(leftURL()), SLOT(urlLeft()) );
    
    clear();
    
    foreach (QLabel *l, m_widget->findChildren<QLabel*>())
        l->setTextInteractionFlags( Qt::TextSelectableByMouse );
      
    #ifdef Q_WS_MAC
    // for stationary watermark painted on viewport()
    foreach (QWidget* w, m_scrollArea->findChildren<QWidget*>())
        w->setAutoFillBackground( false );
    #endif
     
//////
    #ifdef Q_WS_MAC
    // fixes OSX 2-finger trackpad being too fast
    // but makes scroll-wheel speed soooo slow :(
    m_scrollArea->verticalScrollBar()->setSingleStep( 1 );
    #endif
}

void
MetaDataExtension::applyMarginAndSpacing()
{
    int xImage = k_marginLeft;
    int xText = k_marginLeft + ui.albumPic->width() + k_column;

    int yImageUpper = k_marginTop;
    int yTextUpper = k_marginTop;

    int yImageLower = k_marginTop + ui.albumPic->height() + k_row;
    int yTextLower = k_marginTop + ui.albumPic->height() + k_row - 4;

    int xImgOffset = xImage - ui.albumPic->geometry().x();
    int yImgUpperOffset = yImageUpper - ui.albumPic->geometry().y();
    translateItem( ui.albumPic, xImgOffset, yImgUpperOffset );

    int yImgLowerOffset = yImageLower - ui.artistPic->geometry().y();
    translateItem( ui.artistPic, xImgOffset, yImgLowerOffset );

    int xTextOffset = xText - ui.trackLabel->geometry().x();
    int yUpperTextOffset = yTextUpper - ui.trackLabel->geometry().y();
    translateItem( ui.trackLabel, xTextOffset, yUpperTextOffset );
    translateItem( ui.byCaption, xTextOffset, yUpperTextOffset );
    translateItem( ui.artistLabel, xTextOffset, yUpperTextOffset );
    translateItem( ui.buyTrackIcon, xTextOffset, yUpperTextOffset );
    translateItem( ui.buyTrackLabel, xTextOffset, yUpperTextOffset );
    translateItem( ui.albumLabel, xTextOffset, yUpperTextOffset );
    translateItem( ui.releasedCaption, xTextOffset, yUpperTextOffset );
    translateItem( ui.dateAndLabelLabel, xTextOffset, yUpperTextOffset );
    translateItem( ui.totalCaption, xTextOffset, yUpperTextOffset );
    translateItem( ui.numTracksLabel, xTextOffset, yUpperTextOffset );
    translateItem( ui.buyAlbumIcon, xTextOffset, yUpperTextOffset );
    translateItem( ui.buyAlbumLabel, xTextOffset, yUpperTextOffset );

    int yLowerTextOffset = yTextLower - ui.aboutLabel->geometry().y();
    translateItem( ui.aboutLabel, xTextOffset, yLowerTextOffset );
    translateItem( ui.numListenersLabel, xTextOffset, yLowerTextOffset );
    translateItem( ui.wikiLabel, xTextOffset, yLowerTextOffset );
    translateItem( ui.writeWikiButton, xTextOffset, yLowerTextOffset );
    translateItem( ui.wikiLink, xTextOffset, yLowerTextOffset );
    translateItem( ui.tagsLabel, xTextOffset, yLowerTextOffset );
    translateItem( ui.tagNoTagsLabel, xTextOffset, yLowerTextOffset );
    translateItem( ui.tagButton, xTextOffset, yLowerTextOffset );
    translateItem( ui.tagLink, xTextOffset, yLowerTextOffset );
    translateItem( ui.similarLabel, xTextOffset, yLowerTextOffset );
    translateItem( ui.topFansLabel, xTextOffset, yLowerTextOffset );

    m_wikiLinkReadPos = ui.wikiLink->pos();
    m_wikiLinkWritePos = ui.wikiLink->pos() + QPoint( 25, 0 );

    // Set captions to their actual size and move succeeding labels to their
    // correct x pos (changes with language)
    adjustLabelWidth( ui.byCaption, -1, false );
    adjustLabelWidth( ui.releasedCaption, -1, false );
    adjustLabelWidth( ui.totalCaption, -1, false );
    
    if ( ui.byCaption->text().isEmpty() )
    {
        // Some languages can't have a "by", they just write the artist name
        ui.artistLabel->move(
            ui.trackLabel->geometry().x(),
            ui.artistLabel->geometry().y() );
    }
    else
    {
        ui.artistLabel->move(
            ui.byCaption->geometry().right() + ( k_spaceAfterCaption / 2 ),
            ui.artistLabel->geometry().y() );
    }
    ui.dateAndLabelLabel->move(
        ui.releasedCaption->geometry().right() + k_spaceAfterCaption,
        ui.dateAndLabelLabel->geometry().y() );
    ui.numTracksLabel->move(
        ui.totalCaption->geometry().right() + k_spaceAfterCaption,
        ui.numTracksLabel->geometry().y() );
}

void
MetaDataExtension::translateItem( QWidget* item, int xOffset, int yOffset )
{
    item->move( item->geometry().x() + xOffset, item->geometry().y() + yOffset );
}

void
MetaDataExtension::clear()
{
    clearArtist();
    clearAlbum();
    clearTrack();

    resetRequests();
    m_haveTrack = false;
}

void
MetaDataExtension::clearArtist()
{
    ui.artistLabel->clearText();
    ui.aboutLabel->clearText();
    ui.numListenersLabel->clear();
    ui.wikiLabel->clear();
    ui.writeWikiButton->hide();
    ui.wikiLink->clear();
    ui.tagsLabel->clear();
    ui.tagNoTagsLabel->hide();
    ui.tagLink->hide();
    ui.tagButton->hide();

    ui.similarLabel->clear();
    ui.topFansLabel->clear();

    setDefaultArtistPic();
    ui.artistPic->hide();

    adjustArtistLabels();
}

void
MetaDataExtension::clearAlbum()
{
    ui.albumLabel->clearText();
    ui.releasedCaption->hide();
    ui.dateAndLabelLabel->clear();
    ui.totalCaption->hide();
    ui.numTracksLabel->clear();
    ui.buyAlbumIcon->hide();
    ui.buyAlbumLabel->clear();
    ui.buyAlbumLabel->setURL( "" );
    QCursor cursor( Qt::ArrowCursor );
    ui.buyAlbumLabel->setUseCursor( true, &cursor );

    //ui.byCaption->hide();
    //QPixmap p;
    //ui.albumPic->setImages( p, p, p, p );
    setDefaultAlbumCover();
    
    adjustTrackLabels();
}

void
MetaDataExtension::clearTrack()
{
    ui.trackLabel->clearText();
    ui.buyTrackIcon->hide();
    ui.buyTrackLabel->clear();
    ui.buyTrackLabel->setURL( "" );
    QCursor cursor( Qt::ArrowCursor );
    ui.buyTrackLabel->setUseCursor( true, &cursor );
    
    adjustTrackLabels();
}

void
MetaDataExtension::resetRequests()
{
    if (m_artistMetaDataRequest)
        m_artistMetaDataRequest->abort();
    if (m_trackMetaDataRequest)
        m_trackMetaDataRequest->abort();

    m_coverArtistLoader->abort();
    m_coverAlbumLoader->abort();

    m_requestsReset = true;
}

void
MetaDataExtension::setDefaultAlbumCover()
{
    QPixmap image;
    image.load( dataPath( "no_cover.gif" ) );
    if (!image.isNull())
        image = image.scaled( 137, 137, Qt::KeepAspectRatio, Qt::SmoothTransformation );
    ui.albumPic->setImages( image, image, image, image );
}


void
MetaDataExtension::setDefaultArtistPic()
{
    QPixmap image;
    image.load( dataPath( "no_artist.gif" ) );
    image = image.scaled( 137, 137, Qt::KeepAspectRatio, Qt::SmoothTransformation );
    ui.artistPic->setImages( image, image, image, image );
    ui.artistPic->show();
}


void
MetaDataExtension::displayTuningIn()
{
    ui_notPlaying.spinnerLabel->setVisible( true );
    m_stack.setCurrentIndex( 1 );

    ui_notPlaying.messageLabel->setText( tr( "Starting station" ) );

    m_tuning_in_timer->start();
}


void
MetaDataExtension::onDelayedTuningIn()
{
    ui_notPlaying.messageLabel->setText( tr( "Starting station (any second now)" ) );
}


void
MetaDataExtension::setMetaData( MetaData metaData )
{
    // EJ TODO: do we really need mutexes in here any longer?
    QMutexLocker lock( &m_metadataMutex );

    m_metaData = metaData;

    // Only clear artist fields if the artist differs to avoid flicker
    QString newText = m_metaData.artist().toLower();
    QString oldText = ui.artistLabel->text().toLower();
    if ( newText != oldText )
    {
        // TODO: this still flickers in the case of moderation
        clearArtist();
    }

    // Only clear album fields if the album differs to avoid flicker
    newText = m_metaData.album().toLower();
    oldText = ui.albumLabel->text().toLower();
    if ( newText != oldText )
    {
        // TODO: this still flickers in the case of moderation
        clearAlbum();
    }

    //m_requestsReset = false;
    clearTrack();
    resetRequests();

    ui.artistLabel->setText( m_metaData.artist() );
    ui.trackLabel->setText( m_metaData.track() );

    // If we're expecting metadata later, don't show album label until then
    // to avoid a disjointed appearance.
    ui.albumLabel->setText( m_metaData.album() );
    ui.albumLabel->hide();
    bool downloadEnabled = The::currentUser().metadataEnabled();
    if ( !downloadEnabled || m_offline )
    {
        ui.albumLabel->show();
    }

    m_haveTrack = !m_metaData.artist().isEmpty() && !m_metaData.track().isEmpty();

    if ( m_haveTrack )
    {
        adjustTopLabels();
        buildHomeBrewedUrls(); 
        buildTooltips();
        updateDragData(); // this is important since DragLabel::setText() clears the drag-data!

        if ( m_alwaysFetchMetaData )
        {
            requestMetaData();
            m_requestsReset = false;
        }
        
        m_stack.setCurrentIndex( 0 );
        ui.byCaption->show();
    }
    else
    {
        ui.aboutLabel->setText( "" );
        ui.byCaption->hide();
        //emit showRestState();
    }

    //qDebug() << "updateUi" << m_metaData.artist() << "-" << m_metaData.track() << " releasing";
}

void
MetaDataExtension::adjustTopLabels()
{
    adjustLabelWidth( ui.trackLabel );
    int y = adjustLabelWidth( ui.artistLabel );
    y += 17;

    y = adjustLabelWidth( ui.albumLabel, y );

    adjustWidgetSize();
}

void
MetaDataExtension::adjustTrackLabels()
{
    adjustLabelWidth( ui.trackLabel );
    int y = adjustLabelWidth( ui.artistLabel );

    if ( !ui.buyTrackLabel->text().isEmpty() )
    {
        y = adjustLabelWidth( ui.buyTrackLabel );
        moveLabel( ui.buyTrackIcon, y - 2 );
    }
    y += 17;

    y = adjustLabelWidth( ui.albumLabel, y );
    y += 1;

    if ( !ui.dateAndLabelLabel->text().isEmpty() )
    {
        y = adjustLabelWidth( ui.dateAndLabelLabel, y );
        moveLabel( ui.releasedCaption, y );
    }

    if ( !ui.numTracksLabel->text().isEmpty() )
    {
        y += 0;
        y = adjustLabelWidth( ui.numTracksLabel, y );
        moveLabel( ui.totalCaption, y );
    }
    
    if ( !ui.buyAlbumLabel->text().isEmpty() )
    {
        y += 0;
        y = adjustLabelWidth( ui.buyAlbumLabel, y );
        moveLabel( ui.buyAlbumIcon, y - 2 );
    }

    adjustWidgetSize();
}

void
MetaDataExtension::adjustArtistLabels()
{
    int y = adjustLabelWidth( ui.aboutLabel );
    y += 1;
    
    y = adjustLabelWidth( ui.numListenersLabel, y );
    y += 12;

    if ( !ui.wikiLabel->text().isEmpty() )
    {
        y = adjustLabelSize( ui.wikiLabel, y );
        if ( ui.writeWikiButton->isHidden() )
        {
            y += 2;
        }
        else
        {
            y += 6;
        }
    }
    
    y = adjustLabelWidth( ui.wikiLink, y );
    moveLabel( ui.writeWikiButton, y + 2 );
    y += 10;

    if ( !ui.writeWikiButton->isHidden() )
    {
        y += 4;
    }

    if ( !ui.tagsLabel->isHidden() )
    {
        y = adjustLabelSize( ui.tagsLabel, y );
    }
    else
    {
        y = adjustLabelWidth( ui.tagNoTagsLabel, y );
        y += 4;
        adjustLabelWidth( ui.tagButton, y );
        y = adjustLabelWidth( ui.tagLink, y + 4 );
        y += 4;
    }
    y += 10;
    
    if ( !ui.similarLabel->text().isEmpty() )
    {
        y = adjustLabelSize( ui.similarLabel, y );
        y += 10;
    }
    
    if ( !ui.topFansLabel->text().isEmpty() )
    {
        y = adjustLabelSize( ui.topFansLabel, y );
    }

    adjustWidgetSize();
}

int
MetaDataExtension::adjustLabelWidth( QWidget* label, int y, bool adaptToViewport )
{
    QRect geom = label->geometry();

    int neededWidth = label->sizeHint().width();

    if ( adaptToViewport )
    {
        int availWidth = viewWidth() - k_marginLeft - ui.albumPic->width() -
            k_column - k_marginRight;
        availWidth = qMax( 0, availWidth );

        neededWidth = qMin( neededWidth, availWidth );
    }

    geom.setWidth( neededWidth );

    if ( y != -1 )
    {
        geom.moveTop( y );
    }
    
    label->setGeometry( geom );

    return geom.bottom();
}

int
MetaDataExtension::adjustLabelSize( QWidget* label, int y )
{
    QRect geom = label->geometry();

    int availWidth = viewWidth() - k_marginLeft - ui.albumPic->width() -
        k_column - k_marginRight;
    availWidth = qMax( 0, availWidth );
    geom.setWidth( availWidth );
    
    int height = label->heightForWidth( availWidth );
    geom.setHeight( height );

    if ( y != -1 )
    {
        geom.moveTop( y );
    }
    
    label->setGeometry( geom );

    return geom.bottom();
}

void
MetaDataExtension::moveLabel( QWidget* label, int y )
{
    QRect geom = label->geometry();
    geom.moveBottom( y );
    label->setGeometry( geom );
}

void
MetaDataExtension::adjustWidgetSize()
{
    int widest = 0;
    int w = 0;
    if ( ( w = ui.trackLabel->width() ) > widest ) widest = w;
    if ( ( w = ui.artistLabel->width() ) > widest ) widest = w;
    if ( ( w = ui.buyTrackLabel->width() ) > widest ) widest = w;
    if ( ( w = ui.albumLabel->width() ) > widest ) widest = w;
    if ( ( w = ui.dateAndLabelLabel->width() ) > widest ) widest = w;
    if ( ( w = ui.numTracksLabel->width() ) > widest ) widest = w;
    if ( ( w = ui.buyAlbumLabel->width() ) > widest ) widest = w;
    if ( ( w = ui.aboutLabel->width() ) > widest ) widest = w;
    if ( ( w = ui.numListenersLabel->width() ) > widest ) widest = w;
    if ( ( w = ui.wikiLabel->width() ) > widest ) widest = w;
    if ( ( w = ui.wikiLink->width() ) > widest ) widest = w;
    if ( ( w = ui.tagsLabel->width() ) > widest ) widest = w;
    if ( ( w = ui.similarLabel->width() ) > widest ) widest = w;
    if ( ( w = ui.topFansLabel->width() ) > widest ) widest = w;

    int lowest = 0;
    if ( !ui.topFansLabel->text().isEmpty() )
        lowest = ui.topFansLabel->geometry().bottom();
    else if ( !ui.similarLabel->text().isEmpty() )
        lowest = ui.similarLabel->geometry().bottom();
    else if ( !ui.tagsLabel->text().isEmpty() )
        lowest = ui.tagsLabel->geometry().bottom();
    else if ( !ui.wikiLink->text().isEmpty() )
        lowest = ui.wikiLink->geometry().bottom();
    else if ( !ui.wikiLabel->text().isEmpty() )
        lowest = ui.wikiLabel->geometry().bottom();
    else if ( !ui.artistPic->isVisible() )
        lowest = ui.artistPic->geometry().bottom();
    else if ( !ui.numListenersLabel->text().isEmpty() )
        lowest = ui.numListenersLabel->geometry().bottom();
    else if ( !ui.aboutLabel->text().isEmpty() )
        lowest = ui.aboutLabel->geometry().bottom();
    else if ( !ui.buyAlbumLabel->text().isEmpty() )
        lowest = ui.buyAlbumLabel->geometry().bottom();
    else if ( !ui.numTracksLabel->text().isEmpty() )
        lowest = ui.numTracksLabel->geometry().bottom();
    else if ( !ui.dateAndLabelLabel->text().isEmpty() )
        lowest = ui.dateAndLabelLabel->geometry().bottom();
    else if ( !ui.albumLabel->text().isEmpty() )
        lowest = ui.albumLabel->geometry().bottom();
    else if ( !ui.buyTrackLabel->text().isEmpty() )
        lowest = ui.buyTrackLabel->geometry().bottom();
    else if ( !ui.artistLabel->text().isEmpty() )
        lowest = ui.artistLabel->geometry().bottom();
    else if ( !ui.trackLabel->text().isEmpty() )
        lowest = ui.trackLabel->geometry().bottom();

    int width = ( k_marginLeft + k_marginRight ) + ui.albumPic->width() + k_column + widest;
    int height = lowest + k_marginBottom;

    m_widget->setMinimumSize( width, height );
}

int
MetaDataExtension::viewWidth()
{
    int viewWidth = m_scrollArea->viewport()->width();

    QScrollBar* vertBar = m_scrollArea->verticalScrollBar();
    if ( vertBar != NULL )
    {
        // Hmm, this returns the wrong value if we haven't had a scrollbar
        // on screen yet. Hardcode for now.
        //viewWidth -= vertBar->width();
        viewWidth -= 17;
    }

    return viewWidth;
}

void
MetaDataExtension::resizeEvent( QResizeEvent* /*event*/ )
{
    adjustTrackLabels();
    adjustArtistLabels();
}

void
MetaDataExtension::buildHomeBrewedUrls()
{
    QUrl url( m_urlBase );
    if ( m_metaData.artistPageUrl().isEmpty() )
    {
        url.setPath( "/music/" + CUtils::UrlEncodeItem( m_metaData.artist() ) );
        m_metaData.setArtistPageUrl( url.toString() );
    }
    ui.artistLabel->setURL( m_metaData.artistPageUrl() );
    ui.aboutLabel->setURL( m_metaData.artistPageUrl() );

    if ( m_metaData.trackPageUrl().isEmpty() )
    {
        url.setPath( "/music/" +
                     CUtils::UrlEncodeItem( m_metaData.artist() ) + "/_/" +
                     CUtils::UrlEncodeItem( m_metaData.track() ) );
        m_metaData.setTrackPageUrl( url.toString() );
    }
    ui.trackLabel->setURL( m_metaData.trackPageUrl() );

    if ( m_metaData.albumPageUrl().isEmpty() )
    {
        url.setPath( "/music/" +
                     CUtils::UrlEncodeItem( m_metaData.artist() ) + "/" +
                     CUtils::UrlEncodeItem( m_metaData.album() ) );
        m_metaData.setAlbumPageUrl( url.toString() );
    }
    ui.albumLabel->setURL( m_metaData.albumPageUrl() );
}

void
MetaDataExtension::buildTooltips()
{
    QString tipBase = tr( "Drag to tag/recommend %1" );
    QString tipBaseTitle = tr( "Drag to tag/recommend \"%1\"" );
    QString tip;

    QString track = ui.trackLabel->text();
    if ( !track.isEmpty() )
    {
        tip = tipBaseTitle.arg( track );
        ui.trackLabel->setItemTooltip( 0, tip );
    }

    QString artist = ui.artistLabel->text();
    if ( !artist.isEmpty() )
    {
        tip = tipBase.arg( artist );
        ui.artistLabel->setItemTooltip( 0, tip );
        ui.aboutLabel->setItemTooltip( 0, tip );
    }

    QString album = ui.albumLabel->text();
    if ( !album.isEmpty() )
    {
        tip = tipBaseTitle.arg( album );
        ui.albumLabel->setItemTooltip( 0, tip );
    }

    QStringList similar = ui.similarLabel->items();
    for ( int i = 0; i < similar.count(); ++i )
    {
        QString entry = similar.at( i );
        tip = tipBase.arg( entry );
        ui.similarLabel->setItemTooltip( i, tip );
    }
}

void
MetaDataExtension::updateDragData()
{
    // TODO: need to include line-wrapping draglabels too    
    QHash<QString, QString> data;
    data.insert( "artist", m_metaData.artist() );
    data.insert( "album", m_metaData.album() );
    data.insert( "track", m_metaData.track() );

    ui.artistLabel->setItemData( 0, data );
    ui.aboutLabel->setItemData( 0, data );
    ui.albumLabel->setItemData( 0, data );
    ui.trackLabel->setItemData( 0, data );
}

void
MetaDataExtension::requestMetaData()
{
    // If this is a radio track, we will have:
    //  - artist, track, album, albumPicUrl, artistPageUrl, trackPageUrl, albumPageUrl
    // If it's scrobbled, we'll have:
    //  - artist, track, duration
    // Does it matter? No. We have to query service anyway. But if it's a radio track,
    // we could start downloading the cover straight away.

    //qDebug() << "requestMetaData";
    
    //m_metadataFetched = false;
    m_metadataFetched = NothingFetched;

    bool downloadEnabled = The::settings().currentUser().metadataEnabled();
    if ( !m_haveTrack || !downloadEnabled )
    {
        if ( m_alwaysFetchMetaData )
        {
            emit metadataFetched( m_metaData );
        }
        
        return;
    }

    m_requestsReset = false;

    //qDebug() << "requestMetadata" << m_metaData.artist() << "-" << m_metaData.track() << " grabbing";
    
    if (m_trackMetaDataRequest)
        m_trackMetaDataRequest->abort();
    if (m_artistMetaDataRequest)
        m_artistMetaDataRequest->abort();
    
    m_artistMetaDataRequest = new ArtistMetaDataRequest;
    m_trackMetaDataRequest = new TrackMetaDataRequest;
    
    connect( m_artistMetaDataRequest, SIGNAL(result( Request* )), SLOT(artistMetaDataReady()) );
    connect( m_trackMetaDataRequest, SIGNAL(result( Request* )), SLOT(trackMetaDataReady()) );
    
    m_artistMetaDataRequest->setArtist( m_metaData.artist() );
    m_trackMetaDataRequest->setTrack( m_metaData );
    
    m_artistMetaDataRequest->start();
    m_trackMetaDataRequest->start();
}


void
MetaDataExtension::artistMetaDataReady()
{
    Q_ASSERT( m_artistMetaDataRequest );
    
    if (m_artistMetaDataRequest->failed()) {
        //TODO mxcl error message
        //checkMetaDataFetched( ArtistPicFetched );
        return;
    }
    
    MetaData artist = m_artistMetaDataRequest->metaData();
    
    QMutexLocker lock( &m_metadataMutex );

    // Copy new stuff into our own m_metaData (can't just assign as it will
    // overwrite track fields in m_metaData, even more evidence that these
    // need splitting into Track and Artist classes)
    m_metaData.setArtist( artist.artist() );
    m_metaData.setArtistPicUrl( artist.artistPicUrl() );

    // This extra check is needed to not lose the home made URL for the image
    // clicks, as it just queries m_metaData.
    if ( !artist.artistPageUrl().isEmpty() ) m_metaData.setArtistPageUrl( artist.artistPageUrl() );

    m_metaData.setNumListeners( artist.numListeners() );
    m_metaData.setNumPlays( artist.numPlays() );
    m_metaData.setWiki( artist.wiki() );
    m_metaData.setWikiPageUrl( artist.wikiPageUrl() );
    m_metaData.setArtistTags( artist.artistTags() );
    m_metaData.setSimilarArtists( artist.similarArtists() );
    m_metaData.setTopFans( artist.topFans() );

    // Populate artist fields
    ui.artistLabel->setText( m_metaData.artist() );
    ui.aboutLabel->setText( m_metaData.artist() );

    if ( !m_metaData.artistPageUrl().isEmpty() )
    {
        ui.artistLabel->setURL( m_metaData.artistPageUrl() );
        ui.aboutLabel->setURL( m_metaData.artistPageUrl() );
    }
    
    if ( qApp->arguments().contains( "--sanity" ) )
    {
        ui.numListenersLabel->setText(
            tr( "%L1 listeners on Last.fm" ).arg( m_metaData.numListeners() ) );
    }
    else
    {
        ui.numListenersLabel->setText(
            tr( "%L1 plays scrobbled on Last.fm" ).arg( m_metaData.numPlays() ) );
    }

    populateWiki();

    // Have to set defFont on every item or it won't work properly on the Mac
    QFont defFont = m_widget->font();
    defFont.setPixelSize( k_standardFontSize );
    if ( m_metaData.artistTags().isEmpty() )
    {
        ui.tagNoTagsLabel->show();
        ui.tagButton->show();
        ui.tagLink->show();
        
        ui.tagsLabel->hide();
    }
    else
    {
        ui.tagsLabel->setItems( m_metaData.artistTags().mid( 0, 5 ) );
        for ( int i = 0; i < ui.tagsLabel->items().count(); i++ )
        {
            ui.tagsLabel->setItemURL( i, m_urlBase + "/tag/" +
                CUtils::UrlEncodeItem( ui.tagsLabel->items().at( i ) ) );
            ui.tagsLabel->setItemFont( i, defFont );
        }
        ui.tagsLabel->show();
    }
    
    ui.similarLabel->setItems( m_metaData.similarArtists().mid( 0, 5 ) );
    for ( int i = 0; i < ui.similarLabel->items().count(); i++ )
    {
        ui.similarLabel->setItemURL( i, m_urlBase + "/music/" +
            CUtils::UrlEncodeItem( ui.similarLabel->items().at( i ) ) );
        ui.similarLabel->setItemFont( i, defFont );
    }

    ui.topFansLabel->setItems( m_metaData.topFans().mid( 0, 5 ) );
    for ( int i = 0; i < ui.topFansLabel->items().count(); i++ )
    {
        ui.topFansLabel->setItemURL( i, m_urlBase + "/user/" +
            CUtils::UrlEncodeItem( ui.topFansLabel->items().at( i ) ) );
        ui.topFansLabel->setItemFont( i, defFont );
    }

    // No artist pics in Japanese version
    #ifndef HIDE_RADIO
        // Download artist picture
        QUrl url = m_metaData.artistPicUrl();
        if ( !url.isEmpty() )
        {
            downloadPic( m_coverArtistLoader, url );
        }
        else
        {
            setDefaultArtistPic();
            checkMetaDataFetched( ArtistPicFetched );
        }
    #endif

    updateDragData();
    adjustArtistLabels();

    // Need to do this here too as this is included in adjustTrackLabels only
    // since it belongs to the track section.
    adjustLabelWidth( ui.artistLabel );

    buildTooltips();

    //qDebug() << "setMetaDataArtist" << artist.artist() << " releasing";
}


void
MetaDataExtension::trackMetaDataReady()
{
    Q_ASSERT( m_trackMetaDataRequest );

    if (m_trackMetaDataRequest->failed()) {
        //TODO mxcl error message
        //checkMetaDataFetched( AlbumPicFetched );
        return;
    }

    ui.albumLabel->show();
    
    MetaData track = m_trackMetaDataRequest->metaData();

    //qDebug() << "setMetaDataTrack" << track.track() << " grabbing";

    QMutexLocker lock( &m_metadataMutex );

    // Copy new stuff into our own m_metaData
    m_metaData.setArtist( track.artist() );
    m_metaData.setTrack( track.track() );
    m_metaData.setTrackPageUrl( track.trackPageUrl() );
    m_metaData.setAlbum( track.album() );

    // This extra check is needed to not lose the home made URL for the image
    // clicks, as it just queries m_metaData.   
    if ( !track.albumPageUrl().isEmpty() ) m_metaData.setAlbumPageUrl( track.albumPageUrl() );

    m_metaData.setAlbumPicUrl( track.albumPicUrl() );
    m_metaData.setLabel( track.label() );
    m_metaData.setNumTracks( track.numTracks() );
    m_metaData.setReleaseDate( track.releaseDate() );
    m_metaData.setBuyTrackString( track.buyTrackString() );
    m_metaData.setBuyTrackUrl( track.buyTrackUrl() );
    m_metaData.setBuyAlbumString( track.buyAlbumString() );
    m_metaData.setBuyAlbumUrl( track.buyAlbumUrl() );

    // Populate track fields
    ui.trackLabel->setText( m_metaData.track() );
    if ( !m_metaData.trackPageUrl().isEmpty() )
    {
        ui.trackLabel->setURL( m_metaData.trackPageUrl() );
    }

    if ( m_metaData.isTrackBuyable() )
    {
        ui.buyTrackLabel->setText( track.buyTrackString() );
        ui.buyTrackLabel->setURL( track.buyTrackUrl() );
        QCursor cursor( Qt::PointingHandCursor );
        ui.buyTrackLabel->setUseCursor( true, &cursor );
        ui.buyTrackLabel->show();
        ui.buyTrackIcon->show();    
    }

    // Populate album fields
    if ( !m_metaData.album().isEmpty() )
    {
        ui.albumLabel->setText( m_metaData.album() );
    }   
    if ( !m_metaData.albumPageUrl().isEmpty() )
    {
        ui.albumLabel->setURL( m_metaData.albumPageUrl() );
    }
    if ( m_metaData.isAlbumBuyable() )
    {
        ui.buyAlbumLabel->setText( track.buyAlbumString() );
        ui.buyAlbumLabel->setURL( track.buyAlbumUrl() );
        QCursor cursor( Qt::PointingHandCursor );
        ui.buyAlbumLabel->setUseCursor( true, &cursor );
        ui.buyAlbumLabel->show();
        ui.buyAlbumIcon->show();    
    }

    QDate date = m_metaData.releaseDate();
    if ( date.isValid() )
    {
        QString labelStr = date.toString( "d MMM yyyy" );
        if ( !m_metaData.label().isEmpty() )
        {
            labelStr += QString( ", " ) +
                #ifdef Q_WS_MAC
                    QChar( 8471 ) + QString( " " ) +
                #else
                    tr( "on " ) +
                #endif
                    m_metaData.label();
        }
        
        ui.dateAndLabelLabel->setText( labelStr );
        ui.dateAndLabelLabel->show();
        ui.releasedCaption->show();
    }

    if ( m_metaData.numTracks() > 0 )
    {
        ui.numTracksLabel->setText( tr( "%1 tracks" ).arg( m_metaData.numTracks() ) );
        ui.numTracksLabel->show();
        ui.totalCaption->show();
    }

    // Download album cover
    QUrl url = m_metaData.albumPicUrl();
    
    if ( !url.isEmpty() )
    {
        downloadPic( m_coverAlbumLoader, url );
    }
    else
    {
        setDefaultAlbumCover();
        checkMetaDataFetched( AlbumPicFetched );
    }

    updateDragData();
    adjustTrackLabels();
    buildTooltips();
    
    emit trackModerated( m_metaData );

    //qDebug() << "setMetaDataTrack" << track.track() << " releasing";
}

void
MetaDataExtension::populateWiki()
{
    QString wikiText = m_metaData.wiki();

    QString linkText;
    if ( m_metaData.wiki().isEmpty() )
    {
        wikiText = tr( "We don't have a description for this artist yet, "
            "care to help?" );
        linkText = tr( "Write an artist description..." );
        ui.writeWikiButton->show();
        ui.wikiLink->move( m_wikiLinkWritePos );
        ui.wikiLink->setHighlightedColor( k_userActionFontColour );
    }
    else
    {
        linkText = tr( "Read more..." );
        ui.writeWikiButton->hide();
        ui.wikiLink->move( m_wikiLinkReadPos );
        ui.wikiLink->setHighlightedColor( QColor( 0, 0, 0 ) );
    }

    ui.wikiLabel->setText( wikiText );
    ui.wikiLink->setText( linkText );
    ui.wikiLink->setURL( m_metaData.wikiPageUrl() );
}

void
MetaDataExtension::downloadPic( Http* loader, const QUrl& url )
{
    loader->abort(); // no dataAvailable for lingering requests will be sent
    
    loader->setHost( url.host() );
    if ( !url.encodedQuery().isEmpty() )
    {
        loader->get( url.path() + "?" + url.encodedQuery(), true );
    }
    else
    {
        loader->get( url.path(), true );
    }
}

void
MetaDataExtension::coverArtistLoaded( const QByteArray& to )
{
//    renderWithDropShadow( to, ui.artistPic );
    render( to, ui.artistPic );
    ui.artistPic->show();
    checkMetaDataFetched( ArtistPicFetched );
}

void
MetaDataExtension::coverAlbumLoaded( const QByteArray& to )
{
//    renderWithDropShadow( to, ui.albumPic );
    render( to, ui.albumPic );
    checkMetaDataFetched( AlbumPicFetched );
}

void
MetaDataExtension::renderWithDropShadow( const QByteArray& from, ImageButton* to )
{
    if ( from.size() > 0 )
    {
        QPixmap shadowed( 137, 137 ); // size of shadow pic

        QPixmap cover;
        bool success = cover.loadFromData( from );
        
        if ( !success )
        {
            LOGL( 1, "Loading of image from byte array failed." );
        }
        else
        {
            cover = cover.scaled( 128, 128, Qt::KeepAspectRatio, Qt::SmoothTransformation );

            const uint sw = (uint)( cover.width() / 100.0f * 7.0f + 0.5f );
            const uint sh = (uint)( cover.height() / 100.0f * 7.0f + 0.5f );
            shadowed.load( dataPath( "shadow_artist.png" ) );
            shadowed = shadowed.scaled( cover.width() + sw, cover.height() + sh,
                Qt::IgnoreAspectRatio, Qt::SmoothTransformation );
                
            QPainter painter( &shadowed );
            painter.drawPixmap( 1, 1, cover );

            // The above calculation isn't exact so let's draw the border
            // to ensure it's always visible.
            painter.setPen( QColor( 38, 38, 38 ) );
            painter.drawRect( 0, 0, cover.width() + 1, cover.height() + 1 );

            to->setImages( shadowed, shadowed, shadowed, shadowed );
        }
    }
}

void
MetaDataExtension::render( const QByteArray& from, ImageButton* to )
{
    if ( from.size() > 0 )
    {
        QPixmap cover;
        bool success = cover.loadFromData( from );
        if ( !success )
        {
            LOGL( 1, "Loading of image from byte array failed." );
        }
        else
        {
            cover = cover.scaled( 137, 137, Qt::KeepAspectRatio, Qt::SmoothTransformation );
            to->setImages( cover, cover, cover, cover );
        }
    }
}

void
MetaDataExtension::artistPicClicked()
{
    new BrowserThread( m_metaData.artistPageUrl() );
}

void
MetaDataExtension::albumPicClicked()
{
    new BrowserThread( m_metaData.albumPageUrl() );
}

bool
MetaDataExtension::isVisible()
{
    return ( m_parent->isVisible() && !m_parent->isHidden() );
}

void
MetaDataExtension::urlLeft()
{
    emit urlHovered( "" );
}

// Need to override this in order to work out when the widget becomes
// visible, and when we need to request metadata.
void
MetaDataScrollArea::paintEvent( QPaintEvent * event )
{
  #ifdef Q_WS_MAC
    static QPixmap watermark( dataPath( "watermark.png" ) );
    QRect r = viewport()->rect();
    
    QPainter p( viewport() );
//    p.fillRect( r, Qt::white );
    p.drawPixmap( r.bottomRight() - watermark.rect().bottomRight(), watermark );
    
    // m_requestsReset will only be true if we haven't requested
    // metadata yet.
    if ( m_owner->m_haveTrack && m_owner->m_requestsReset )
    {   
        QMutexLocker lock( &m_owner->m_metadataMutex );
        m_owner->requestMetaData();
    }    
  #else
    QScrollArea::paintEvent( event );
  #endif
}

void
MetaDataWidget::paintEvent( QPaintEvent * event )
{
    WatermarkWidget::paintEvent( event );

    // m_requestsReset will only be true if we haven't requested
    // metadata yet.
    if ( m_owner->m_haveTrack && m_owner->m_requestsReset )
    {   
        QMutexLocker lock( &m_owner->m_metadataMutex );
        m_owner->requestMetaData();
    }
}

void
MetaDataExtension::buyTrackIconClicked()
{
    QDesktopServices::openUrl( m_metaData.buyTrackUrl() );
}

void
MetaDataExtension::buyAlbumIconClicked()
{
    QDesktopServices::openUrl( m_metaData.buyAlbumUrl() );
}

void
MetaDataExtension::checkMetaDataFetched( MetaDataFetchedState state )
{
    #ifndef HIDE_RADIO
        if ( m_metadataFetched == NothingFetched )
        {
            m_metadataFetched = state;
        }
        else if ( ( m_metadataFetched == AlbumPicFetched || m_metadataFetched == ArtistPicFetched ) && m_metadataFetched != state )
        {
            emit metadataFetched( m_metaData );
            m_metadataFetched = NothingFetched;
        }
    #else
        // Emit after loading the album cover in Japanese version
        if ( state == AlbumPicFetched ) 
            emit metadataFetched( m_metaData );
    #endif
}
