/***************************************************************************
 *   Copyright (C) 2005 - 2007 by                                          *
 *      Christian Muehlhaeuser, Last.fm Ltd <chris@last.fm>                *
 *      Erik Jaelevik, Last.fm Ltd <erik@last.fm>                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02111-1307, USA.          *
 ***************************************************************************/

#include <QtGui>

#ifdef WIN32
    #include <windows.h>
#endif

#include "containerutils.h"
#include <QHeaderView>
#include "Radio.h"
#include "Settings.h"
#include "settingsdialog.h"
#include "lastfmapplication.h"
#include "lastfm_common.h"
#include "container.h"
#include "LastMessageBox.h"


// This is a terrible way of entering strings but I couldn't get VS to recognise
// the Unicode chars in any other way.
static const unsigned char kChinese[] =
    { 0xE4, 0xB8, 0xAD, 0xE6, 0x96, 0x87, 0x0 };
static const unsigned char kRussian[] =
    { 0xD0, 0xA0, 0xD1, 0x83, 0xD1, 0x81, 0xD1, 0x81, 0xD0, 0xBA,
      0xD0, 0xB8, 0xD0, 0xB9, 0x0 };
static const unsigned char kJapanese[] =
    { 0xE6, 0x97, 0xA5, 0xE6, 0x9C, 0xAC, 0xE8, 0xAA, 0x9E, 0x0 }; 

SettingsDialog::SettingsDialog( QWidget *parent )
        : QDialog( parent )
{
    ui.setupUi( this );

    #ifdef Q_WS_MAC
        setWindowTitle( tr( "Last.fm Preferences" ) );
    #else
        setWindowTitle( tr( "Last.fm Options" ) );
    #endif

    // Create pages and add them to stack widget
    QWidget* accountWidget = new QWidget();
    ui_account.setupUi( accountWidget );
    ui.pageStack->addWidget( accountWidget );

    QWidget* radioWidget = new QWidget();
    ui_radio.setupUi( radioWidget );
    ui.pageStack->addWidget( radioWidget );

    QWidget* scrobWidget = new QWidget();
    ui_scrobbling.setupUi( scrobWidget );
    ui.pageStack->addWidget( scrobWidget );

    QWidget* connWidget = new QWidget();
    ui_connection.setupUi( connWidget );
    ui.pageStack->addWidget( connWidget );

    QWidget* mediadeviceWidget = new QWidget();
    ui_mediadevices.setupUi( mediadeviceWidget );
    ui_mediadevices.deviceWidget->header()->setResizeMode( QHeaderView::ResizeToContents );
    ui.pageStack->addWidget( mediadeviceWidget );

    #ifndef WIN32
        ui_connection.crashReportCheck->hide();
    #endif

    // Add icons to user icon dropdown
    QPixmap pixmap( dataPath( "icons/user_red.png" ) );
    ui_account.colourCombo->setItemIcon( 0, pixmap );
    pixmap.load( dataPath( "icons/user_blue.png" ) );
    ui_account.colourCombo->setItemIcon( 1, pixmap );
    pixmap.load( dataPath( "icons/user_green.png" ) );
    ui_account.colourCombo->setItemIcon( 2, pixmap );
    pixmap.load( dataPath( "icons/user_orange.png" ) );
    ui_account.colourCombo->setItemIcon( 3, pixmap );
    pixmap.load( dataPath( "icons/user_black.png" ) );
    ui_account.colourCombo->setItemIcon( 4, pixmap );

    // Add languages to language drop-down
    ui_account.languageCombo->addItem( "English", qtLanguageToLfmLangCode( QLocale::English ) );
    ui_account.languageCombo->addItem( QString("Fran") + QChar( 0xe7 ) + QString( "ais" ), qtLanguageToLfmLangCode( QLocale::French ) );
    ui_account.languageCombo->addItem( "Italiano", qtLanguageToLfmLangCode( QLocale::Italian ) );
    ui_account.languageCombo->addItem( "Deutsch", qtLanguageToLfmLangCode( QLocale::German ) );
    ui_account.languageCombo->addItem( QString( "Espa" ) + QChar( 0xf1 ) + QString( "ol" ), qtLanguageToLfmLangCode( QLocale::Spanish ) );
    ui_account.languageCombo->addItem( QString( "Portugu" ) + QChar( 0xea ) + QString( "s" ), qtLanguageToLfmLangCode( QLocale::Portuguese ) );
    ui_account.languageCombo->addItem( "Polski", qtLanguageToLfmLangCode( QLocale::Polish ) );
    ui_account.languageCombo->addItem( "Svenska", qtLanguageToLfmLangCode( QLocale::Swedish ) );
	ui_account.languageCombo->addItem( QString::fromUtf8("Türkçe"), qtLanguageToLfmLangCode( QLocale::Turkish ) );
    ui_account.languageCombo->addItem( QString::fromUtf8( (const char*) kRussian ), qtLanguageToLfmLangCode( QLocale::Russian ) );
    ui_account.languageCombo->addItem( QString::fromUtf8( (const char*) kChinese ), qtLanguageToLfmLangCode( QLocale::Chinese ) );

    // Add icons to sidebar
    pixmap.load( dataPath( "/icons/options_account.png" ) );
    //pixmap.scaled( 48, 48, Qt::IgnoreAspectRatio, Qt::SmoothTransformation );
    ui.pageList->item( 0 )->setIcon( pixmap );

    pixmap.load( dataPath( "/icons/options_radio.png" ) );
    //pixmap.scaled( 48, 48, Qt::IgnoreAspectRatio, Qt::SmoothTransformation );
    ui.pageList->item( 1 )->setIcon( pixmap );

    #ifdef HIDE_RADIO
        ui.pageList->setItemHidden( ui.pageList->item( 1 ), true );
    #endif // HIDE_RADIO

    pixmap.load( dataPath( "/icons/options_scrobbling.png" ) );
    //pixmap.scaled( 48, 48, Qt::IgnoreAspectRatio, Qt::SmoothTransformation );
    ui.pageList->item( 2 )->setIcon( pixmap );

    pixmap.load( dataPath( "/icons/options_connection.png" ) );
    //pixmap.scaled( 48, 48, Qt::IgnoreAspectRatio, Qt::SmoothTransformation );
    ui.pageList->item( 3 )->setIcon( pixmap );

    pixmap.load( dataPath( "/icons/options_mediadevices.png" ) );
    //pixmap.scaled( 48, 48, Qt::IgnoreAspectRatio, Qt::SmoothTransformation );
    ui.pageList->item( 4 )->setIcon( pixmap );

    #ifdef HIDE_RADIO
        // Need to disable this as the language choice is hardcoded at startup
        // (see LastfmApp::initTranslator) for the jp version.
        ui_account.languageCombo->hide();
        ui_account.languageLabel->hide();
    #endif

    #ifdef Q_WS_MAC
        ui_account.languageCombo->hide();
        ui_account.languageLabel->hide();
        ui_account.colourCombo->hide();
        ui_account.colourLabel->hide();
	#else
		ui_account.showInDockCheck->hide();
    #endif

    ui_scrobbling.scrobblePointSlider->setMinimum( Constants::kScrobblePointMin );
    ui_scrobbling.scrobblePointSlider->setMaximum( Constants::kScrobblePointMax );
        
    // Fix width of scrobble point label so it doesn't make the slider jump about
    int maxWidth = ui_scrobbling.scrobblePointLabel->fontMetrics().width( "100" );
    ui_scrobbling.scrobblePointLabel->setFixedWidth( maxWidth );

    #ifdef Q_WS_MAC
        ui_scrobbling.launchWithMediaPlayerCheck->setText(
            tr( "Launch Last.fm with iTunes" ) );
    #else
        ui_scrobbling.launchWithMediaPlayerCheck->hide();
    #endif

    connect( ui.pageList, SIGNAL(currentRowChanged( int )), SLOT(pageChanged( int )) );
    connect( ui.applyButton, SIGNAL(clicked()), SLOT(applyPressed()) );
    connect( ui_connection.clearCacheButton, SIGNAL(clicked()), SLOT(clearCache()) );

    connect( ui_account.usernameEdit, SIGNAL( textChanged( QString ) ), this, SLOT( configChanged() ) );
    connect( ui_account.passwordEdit, SIGNAL( textChanged( QString ) ), this, SLOT( configChanged() ) );
    connect( ui_account.rememberPassCheck, SIGNAL( toggled( bool ) ), this, SLOT( configChanged() ) );
    connect( ui_account.colourCombo, SIGNAL( currentIndexChanged( int ) ), this, SLOT( configChanged() ) );
    connect( ui_account.languageCombo, SIGNAL( currentIndexChanged( int ) ), this, SLOT( configChanged() ) );
    connect( ui_radio.resumeCheckBox, SIGNAL( toggled( bool ) ), this, SLOT( configChanged() ) );
    connect( ui_radio.cardBox, SIGNAL( currentIndexChanged( int ) ), this, SLOT( configChanged() ) );
    connect( ui_radio.systemBox, SIGNAL( currentIndexChanged( int ) ), this, SLOT( configChanged() ) );
    connect( ui_radio.automaticBufferCheck, SIGNAL( toggled( bool ) ), this, SLOT( configChanged() ) );
    connect( ui_radio.bufferEdit, SIGNAL( textChanged( QString ) ), this, SLOT( configChanged() ) );
    connect( ui_scrobbling.dirTree, SIGNAL( dataChanged() ), this, SLOT( configChanged() ) );
    connect( ui_scrobbling.scrobblePointSlider, SIGNAL( valueChanged( int ) ), this, SLOT( configChanged() ) );
    connect( ui_scrobbling.launchWithMediaPlayerCheck, SIGNAL( toggled( bool ) ), this, SLOT( configChanged() ) );
    connect( ui_connection.proxyBox, SIGNAL( toggled( bool ) ), this, SLOT( configChanged() ) );
    connect( ui_connection.proxyHostEdit, SIGNAL( textChanged( QString ) ), this, SLOT( configChanged() ) );
    connect( ui_connection.proxyPortEdit, SIGNAL( textChanged( QString ) ), this, SLOT( configChanged() ) );
    connect( ui_connection.proxyUsernameEdit, SIGNAL( textChanged( QString ) ), this, SLOT( configChanged() ) );
    connect( ui_connection.proxyPasswordEdit, SIGNAL( textChanged( QString ) ), this, SLOT( configChanged() ) );
    connect( ui_connection.downloadMetadataCheck, SIGNAL( toggled( bool ) ), this, SLOT( configChanged() ) );
    connect( ui_connection.crashReportCheck, SIGNAL( toggled( bool ) ), this, SLOT( configChanged() ) );
    connect( ui_mediadevices.deviceWidget, SIGNAL( currentItemChanged( QTreeWidgetItem*, QTreeWidgetItem* ) ), this, SLOT( mediaDeviceItemSelected( QTreeWidgetItem*, QTreeWidgetItem* ) ) );
    connect( ui_mediadevices.removeButton, SIGNAL( clicked() ), this, SLOT( removeMediaDevice() ) );
	
	#ifdef Q_WS_MAC
		connect( ui_account.showInDockCheck, SIGNAL( toggled( bool ) ), this, SLOT( configChanged() ) );
	#endif
}

int
SettingsDialog::exec()
{
    originalUsername = The::settings().currentUser().username();
    originalPassword = The::settings().currentUser().password();
    originalProxyHost = The::settings().getProxyHost();
    originalProxyUsername = The::settings().getProxyUser();
    originalProxyPassword = The::settings().getProxyPassword();
    originalProxyPort = The::settings().getProxyPort();
    originalProxyUsage = The::settings().isUseProxy();

    populateAccount();
    populateRadio();
    populateScrobbling();
    populateConnection();
    populateMediaDevices();
    foreach (ExtensionInterface *i, extensions)
        i->populateSettings();

    pageChanged( 0 );
    ui.applyButton->setEnabled( false );
    
    return QDialog::exec();
} 

void
SettingsDialog::populateAccount()
{
    UserSettings* user = &The::settings().currentUser();

    ui_account.usernameEdit->setText( user->username() );
    if (!user->password().isEmpty())
        ui_account.passwordEdit->setText( "********" );
    ui_account.rememberPassCheck->setChecked( user->rememberPass() );

    ui_account.colourCombo->setCurrentIndex( user->icon() < ui_account.colourCombo->count() ? user->icon() : 0 );
	
	#ifdef Q_WS_MAC
		ui_account.showInDockCheck->setChecked( isDockIconVisible() );
	#endif

    #ifndef HIDE_RADIO
        QString langCode = The::settings().appLanguage();
        for( int i = 0; i < ui_account.languageCombo->count(); ++i )
        {
            QString code = ui_account.languageCombo->itemData( i ).toString();
            if ( code == langCode )
            {
                ui_account.languageCombo->setCurrentIndex( i );
            }
        }
    #endif // HIDE_RADIO
}


void
SettingsDialog::populateRadio()
{
    ui_radio.resumeCheckBox->setChecked( The::settings().currentUser().resumePlayback() );

    ui_radio.systemBox->clear();
    ui_radio.cardBox->clear();

    Radio& radio = Container::instance().radio();
    ui_radio.systemBox->addItems( radio.soundSystems() );
    ui_radio.cardBox->addItems( radio.devices() );

    ui_radio.systemBox->setCurrentIndex( The::settings().soundSystem() >= 0 ? The::settings().soundSystem() : 0 );
    ui_radio.cardBox->setCurrentIndex( The::settings().soundCard() >= 0 ? The::settings().soundCard() : 0 );

    ui_radio.automaticBufferCheck->setChecked( The::settings().isBufferManagedAutomatically() );
    ui_radio.bufferEdit->setText( QString::number( The::settings().httpBufferSize() / 1024 ) );
}


void
SettingsDialog::populateScrobbling()
{
    UserSettings& user = The::settings().currentUser();

    ui_scrobbling.scrobblePointSlider->setValue( user.scrobblePoint() );
    ui_scrobbling.scrobblePointLabel->setText( QString::number( user.scrobblePoint() ) );

    ui_scrobbling.launchWithMediaPlayerCheck->setChecked( user.launchWithMediaPlayer() );

    ui_scrobbling.dirTree->setExclusions( user.excludedDirs() );
}


void
SettingsDialog::populateConnection()
{
    ui_connection.proxyBox->setChecked( The::settings().isUseProxy() );
    ui_connection.proxyHostEdit->setText( The::settings().getProxyHost() );
    ui_connection.proxyPortEdit->setText( QString::number( The::settings().getProxyPort() ) );
    ui_connection.proxyUsernameEdit->setText( The::settings().getProxyUser() );
    ui_connection.proxyPasswordEdit->setText( The::settings().getProxyPassword() );
    ui_connection.downloadMetadataCheck->setChecked( The::settings().currentUser().metadataEnabled() );
    ui_connection.crashReportCheck->setChecked( The::settings().currentUser().crashReportingEnabled() );
}


void
SettingsDialog::populateMediaDevices()
{
//     qDebug() << "Populating mediadevices";
    QList<QTreeWidgetItem *> items;

    ui_mediadevices.deviceWidget->clear();

    foreach( QString device, The::settings().allMediaDevices() )
    {
        if ( !The::settings().mediaDeviceUser( device ).isEmpty() )
        {
            QStringList list;
            list << device;
            list << The::settings().mediaDeviceUser( device );

            items.append( new QTreeWidgetItem( (QTreeWidget*)0, list ) );
//             ui_mediadevices.removeButton->setEnabled( true );
        }
    }

    ui_mediadevices.deviceWidget->insertTopLevelItems( 0, items );
}


void
SettingsDialog::pageChanged( int currentRow )
{
    ui.pageList->setCurrentRow( currentRow );
    ui.pageStack->setCurrentIndex( currentRow + 1 );
    m_pagesToSave.insert( currentRow );
}


void
SettingsDialog::applyPressed()
{
    // Do hardcoded sections
    if (m_pagesToSave.contains( 0 )) { saveAccount(); }
    if (m_pagesToSave.contains( 1 )) { saveRadio(); }
    if (m_pagesToSave.contains( 2 )) { saveScrobbling(); }
    if (m_pagesToSave.contains( 3 )) { saveConnection(); }
    if (m_pagesToSave.contains( 4 )) { saveMediaDevices(); }

    for (int i = 0; i < 5; ++i)
        m_pagesToSave.remove( i );

    // Do extensions
    foreach (int page, m_pagesToSave) {
        int idx = page - 5;
        extensions[idx]->saveSettings();
    }

    // allow apply to work if it is pushed again for this page
    m_pagesToSave.insert( ui.pageList->currentRow() );

    ui.applyButton->setEnabled( false );
}


void
SettingsDialog::saveAccount()
{
    The::settings().setCurrentUsername( ui_account.usernameEdit->text() );

    UserSettings& user = The::settings().currentUser();
    int iconIndex = ui_account.colourCombo->currentIndex() >= 0 ? ui_account.colourCombo->currentIndex() : 0;

    user.setPassword( ui_account.passwordEdit->text() );
    user.setIcon( static_cast<UserIconColour>( iconIndex ) );
    user.setRememberPass( ui_account.rememberPassCheck->isChecked() );

    #ifndef HIDE_RADIO
        // Store language in config
        int langIdx = ui_account.languageCombo->currentIndex();
        QString langCode = ui_account.languageCombo->itemData( langIdx ).toString();
        QString oldLang = The::settings().appLanguage();

        if ( langCode != oldLang )
        {
            LastMessageBox dlg( QMessageBox::Information,
                    tr( "Restart needed" ),
                    tr( "You need to restart the application for the language "
                        "change to take effect." ),
                    QMessageBox::Ok );
            dlg.exec();
        }

        The::settings().setAppLanguage( langCode );
    #endif
		
	#ifdef Q_WS_MAC
		toggleDockIcon();
	#endif
}


void
SettingsDialog::saveRadio()
{
    The::settings().currentUser().setResumePlayback( ui_radio.resumeCheckBox->isChecked() );
    The::settings().setSoundCard( ui_radio.cardBox->currentIndex() >= 0 ? ui_radio.cardBox->currentIndex() : 0 );
    The::settings().setSoundSystem( ui_radio.systemBox->currentIndex() >= 0 ? ui_radio.systemBox->currentIndex() : 0 );

    The::settings().setBufferManagedAutomatically( ui_radio.automaticBufferCheck->isChecked() );

    // Returns 0 if the conversion fails so is safe
    int bufSize = ui_radio.bufferEdit->text().toInt() * 1024;
    if ( bufSize <= 0 ) bufSize = Constants::kHttpBufferMinSize;
    The::settings().setHttpBufferSize( bufSize );
}


void
SettingsDialog::saveScrobbling()
{
    UserSettings& user = The::settings().currentUser();
    user.setScrobblePoint( ui_scrobbling.scrobblePointSlider->value() );
    user.setExcludedDirs( ui_scrobbling.dirTree->getExclusions() );
    user.setLaunchWithMediaPlayer( ui_scrobbling.launchWithMediaPlayerCheck->isChecked() );
}


void
SettingsDialog::saveConnection()
{
    The::settings().setProxyUser( ui_connection.proxyUsernameEdit->text() );
    The::settings().setProxyPassword( ui_connection.proxyPasswordEdit->text() );
    The::settings().setProxyHost( ui_connection.proxyHostEdit->text() );
    The::settings().setProxyPort( ui_connection.proxyPortEdit->text().toInt() );
    The::settings().setUseProxy( ui_connection.proxyBox->isChecked() );
    The::settings().currentUser().setMetadataEnabled( ui_connection.downloadMetadataCheck->isChecked() );
    The::settings().currentUser().setCrashReportingEnabled( ui_connection.crashReportCheck->isChecked() );
}


// EJ: Umm, surely this ought to be taken care of by the plugin itself,
// and not coded straight into the Container? Like how the Skype
// plugin does it.
void
SettingsDialog::saveMediaDevices()
{}


void
SettingsDialog::mediaDeviceItemSelected( QTreeWidgetItem* /*current*/, QTreeWidgetItem* /*previous*/ )
{
    ui_mediadevices.removeButton->setEnabled( ui_mediadevices.deviceWidget->currentItem() != 0 );
}


void
SettingsDialog::removeMediaDevice()
{
    if ( ui_mediadevices.deviceWidget->currentItem() )
    {
//         qDebug() << "Removing device:" << ui_mediadevices.deviceWidget->currentItem()->text( 0 );
        The::settings().removeMediaDevice( ui_mediadevices.deviceWidget->currentItem()->text( 0 ) );
        delete ui_mediadevices.deviceWidget->currentItem();
    }

    ui_mediadevices.removeButton->setEnabled( ui_mediadevices.deviceWidget->topLevelItemCount() > 0 );
}


void
SettingsDialog::accept()
{
    applyPressed();
    reject();
    
    QDialog::accept();    
}

void
SettingsDialog::reject()
{
    QDialog::reject();

    // we don't set all settings in apply()
    // so we need to check when cancel is pressed if these settings need to be
    // applied, because the user may have clicked apply, then cancel

    bool reconnect = false;
    bool reaudio = false;

    QString password = ui_account.passwordEdit->text();

    password = (password.isEmpty() || password == "********")
            ? originalPassword
            : MD5Digest( ui_account.passwordEdit->text().toAscii() );

    
    if ( m_pagesToSave.contains( 0 ) )
    {
        reconnect = ui_account.usernameEdit->text() != originalUsername ||
                    password != originalPassword;
    }
    if ( !reconnect && m_pagesToSave.contains( 3 ) )
    {
        reconnect =  ui_connection.proxyHostEdit->text() != originalProxyHost ||
                     ui_connection.proxyUsernameEdit->text() != originalProxyUsername ||
                     ui_connection.proxyPasswordEdit->text() != originalProxyPassword ||
                     ui_connection.proxyPortEdit->text().toInt() != originalProxyPort ||
                     ui_connection.proxyBox->isChecked() != originalProxyUsage;
    }

    if ( m_pagesToSave.contains( 1 ) )
    {
        reaudio = ui_radio.cardBox->currentIndex() != originalSoundCard ||
                   ui_radio.systemBox->currentIndex() != originalSoundSystem;
    }

    The::settings().save( reconnect, reaudio );
}


void
SettingsDialog::configChanged()
{
    ui.applyButton->setEnabled( true );
}

void
SettingsDialog::addExtension( ExtensionInterface* ext )
{
    extensions.append( ext );

    ext->setOwner( this );

    ui.pageStack->addWidget( ext->settingsPane() );
    ui.pageList->addItem( ext->tabCaption() );

    int idx = ui.pageList->count() - 1;
    ui.pageList->item( idx )->setIcon( *ext->settingsIcon() );

    // God knows why I have to give it the name again, but I do
    ui.pageList->item( idx )->setText( ext->tabCaption() );

    connect( ext,  SIGNAL( settingsChanged() ),
             this, SLOT( configChanged() ) );
}

void
SettingsDialog::clearCache()
{
    // Doesn't really belong here but...

    int sure = LastMessageBox::question(
        tr( "Confirm" ),
        tr( "Are you sure you want to delete all\ncached images and bios?" ),
        QMessageBox::Yes,
        QMessageBox::No );

    if ( sure == QMessageBox::Yes )
    {
        QApplication::setOverrideCursor( QCursor( Qt::WaitCursor ) );

        QDir dir( savePath( "cache/" ) );

        QStringList files = dir.entryList( QDir::Files );
        bool success = true;
        int count = 0;
        foreach( QString filename, files )
        {
            bool deleted = dir.remove( filename );
            if ( deleted ) { count++; }
            success &= deleted;

            if ( count % 25 == 0 )
            {
                QApplication::processEvents();
            }
        }

        QApplication::restoreOverrideCursor();

        QString result;
        if ( success )
        {
            result = tr( "Cache emptied. %n file(s) deleted.", "", count );
        }
        else
        {
            result = tr( "Not all items could be deleted. You might need "
                "to delete some items manually from\n'%1'." )
                .arg( dir.absolutePath() );
        }

        LastMessageBox::information(
            tr( "Finished" ),
            result,
            QMessageBox::Ok );

        LOGL( 3, "Emptied cache, files: " << count << ", success: " << success );
    }
}

#ifdef Q_WS_MAC
	void 
	SettingsDialog::toggleDockIcon()
	{		
		if ( ui_account.preferencesGroup->isHidden() )
			return;
		
		// Application itself is in .app/Content/MacOS/, info.plist in .app/Content -> ../info.plist
		#ifdef QT_NO_DEBUG
			QString infoPlistFile = "../info.plist";
		#else
			// It's only possible to test this function in debug mode on a working final release - located usually in /Applications
			// This means that you can't hide the icon of the debug version itself in debug mode.
            // NOTE: There has to be a working release version of the application in the path above! Otherwise this option is invisible...
			QString infoPlistFile = "/Applications/Last.fm.app/Contents/info.plist";
		#endif
		
		QFile file( infoPlistFile );
        
		if ( file.open( QIODevice::ReadOnly ) ) 
		{
			// Open plist and read all
			QTextStream fileStream( &file );
			fileStream.setCodec( "UTF-8" );
			
			QString fileContent = fileStream.readAll();
			
			file.close();
			
			// Determine if plist includes the LSUIElement key, which sets the icon visibility
			bool previousDockIconState;
			
			if ( fileContent.indexOf( "\t<key>LSUIElement</key>\n\t<string>1</string>\n" ) == -1 )
				previousDockIconState = true;
			else
				previousDockIconState = false;
			
			// Check if there are any changes to the saved state in the plist
			if ( previousDockIconState == ui_account.showInDockCheck->isChecked() )
				return;
			
			LastMessageBox dlg( QMessageBox::Information,
							 tr( "Restart needed" ),
							 tr( "You need to restart the application for the dock icon appearance "
								 "change to take effect." ),
							 QMessageBox::Ok );
            dlg.exec();
			
			// Open plist file again, this time with writing access
			if ( file.open( QIODevice::WriteOnly | QIODevice::Truncate ) )
			{
				// Check if icon isn't set hidden and the user wants to hide it
				if ( fileContent.indexOf( "\t<key>LSUIElement</key>\n\t<string>1</string>\n" ) == -1 
											&& !ui_account.showInDockCheck->isChecked() )
				{
					fileContent.replace( "</dict>\n"
										 "</plist>", 
										 "\t<key>LSUIElement</key>\n"
										 "\t<string>1</string>\n"
										 "</dict>\n"
										 "</plist>", Qt::CaseInsensitive );
					fileContent.replace( "\t<key>LSUIElement</key>\n"
										 "\t<string>0</string>\n", 
										 "", Qt::CaseInsensitive);
				}
				else
				{
					fileContent.replace( "\t<key>LSUIElement</key>\n"
										 "\t<string>0</string>\n", "", Qt::CaseInsensitive );
					fileContent.replace( "\t<key>LSUIElement</key>\n"
										 "\t<string>1</string>\n", "", Qt::CaseInsensitive );
				}
				
				fileStream.seek( 0 );
				fileStream << fileContent;
				
				fileStream.flush();
				file.close();
				
				QProcess launcherService;
				launcherService.start( "/System/Library/Frameworks/ApplicationServices.framework"
									// See comment above why I'm doing this...
									#ifdef QT_NO_DEBUG
										"/Versions/A/Frameworks/LaunchServices.framework/Versions/A/Support/lsregister -v -f " +
										QCoreApplication::applicationDirPath().remove( "/Contents/MacOS" ) );
									#else
									   "/Versions/A/Frameworks/LaunchServices.framework/Versions/A/Support/lsregister -v -f /Applications/LastFM.app");
									#endif
				
				if ( !launcherService.waitForFinished() )
					qDebug() << "Error while toggling dock icon.";
				else
					qDebug() << "Toggled dock icon: " << ui_account.showInDockCheck->isChecked();
			}
			else
				qDebug() << "Error while toggling dock icon.";
        }
		else
			qDebug() << "Error while toggling dock icon.";
	}

	bool 
	SettingsDialog::isDockIconVisible()
	{
		// Application itself is in .app/Content/MacOS/, info.plist in .app/Content -> ../info.plist
		#ifdef QT_NO_DEBUG
			QString infoPlistFile = "../info.plist";
		#else
		// It's only possible to test this function in debug mode on a working final release - located usually in /Applications
		// This means that you can't hide the icon of the debug version itself in debug mode.
        // NOTE: There has to be a working release version of the application in the path above! Otherwise this option is invisible...
			QString infoPlistFile = "/Applications/Last.fm.app/Contents/info.plist";
		#endif
		
		QFile file( infoPlistFile );
        
		// Read & WRITE (to check if we have permissions to modify the plist)
		if ( file.open( QIODevice::ReadWrite ) )
		{
			QTextStream fileStream( &file );
			fileStream.setCodec( "UTF-8" );
			
			QString fileContent = fileStream.readAll();
			
			file.close();
			
			if ( fileContent.indexOf( "\t<key>LSUIElement</key>\n\t<string>1</string>\n" ) == -1 )
				return true;
			else
				return false;
		}
		else
		{	 
			qDebug() << "Dock icon toggling is deactivated becaue user isn't admin.";
			ui_account.preferencesGroup->hide();
			return false;
		}
	}
#endif
